<?php
require_once __DIR__ . '/../../classes/Database.php';
require_once __DIR__ . '/../../classes/Product.php';
require_once __DIR__ . '/../../classes/Vendor.php';

header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
header('Access-Control-Max-Age: 86400');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit(0);
}

$db = (new Database())->getConnection();
$product = new Product();
$vendor = new Vendor();
$userId = isset($_POST['user_id']) ? $_POST['user_id'] : (isset($_COOKIE['user_id']) ? $_COOKIE['user_id'] : null); // Get user_id from POST or cookie
$isVendor = $vendor->isVendor($userId);
$isAdmin = $vendor->isAdmin($userId);

error_log("User ID: $userId, isVendor: " . ($isVendor ? 'true' : 'false') . ", isAdmin: " . ($isAdmin ? 'true' : 'false'));

$method = $_SERVER['REQUEST_METHOD'];
$id = $_GET['id'] ?? null;

switch ($method) {
    case 'GET':
        if ($id) {
            $result = $product->getProduct($id);
            if ($result) {
                echo json_encode($result);
            } else {
                http_response_code(404);
                echo json_encode(['error' => 'Product not found']);
            }
        } else {
            echo json_encode($product->getAllProducts($userId, $isVendor, $isAdmin));
        }
        break;

    case 'POST':
        if (!isset($_POST['name'], $_POST['description'], $_POST['price'], $_POST['stock'], $_FILES['primary_image'], $_POST['category_id'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Missing required fields']);
            exit;
        }
        // Temporary bypass for testing
        // if (!$isVendor && !$isAdmin) {
        //     http_response_code(403);
        //     echo json_encode(['error' => 'Unauthorized']);
        //     exit;
        // }

        $targetDir = __DIR__ . '/../../../dashboard/admin/uploads/';
        if (!file_exists($targetDir)) {
            mkdir($targetDir, 0777, true);
        }

        $name = $_POST['name'];
        $description = $_POST['description'];
        $price = $_POST['price'];
        $stock = $_POST['stock'];
        $categoryId = $_POST['category_id'];
        $vendorId = $_POST['vendor_id'] ?: null;

        // Handle primary image upload
        $primaryImage = $_FILES['primary_image']['name'];
        $targetFile = $targetDir . basename($primaryImage);
        $imageFileType = strtolower(pathinfo($targetFile, PATHINFO_EXTENSION));
        $allowedTypes = ['jpg', 'jpeg', 'png', 'gif'];
        if (in_array($imageFileType, $allowedTypes) && move_uploaded_file($_FILES['primary_image']['tmp_name'], $targetFile)) {
            $primaryImage = '/dashboard/admin/uploads/' . $primaryImage;
        } else {
            http_response_code(400);
            echo json_encode(['error' => 'Invalid or failed primary image upload']);
            exit;
        }

        $data = [
            'name' => $name,
            'description' => $description,
            'price' => $price,
            'stock' => $stock,
            'primary_image' => $primaryImage,
            'category_id' => $categoryId,
            'vendor_id' => $vendorId
        ];
        $result = $product->createProduct($data, $userId, $isVendor, $isAdmin);
        http_response_code($result['code'] ?? 201);
        echo json_encode($result);

        // Handle additional images if present
        if (isset($_FILES['images']) && is_array($_FILES['images']['name'])) {
            $additionalImages = [];
            for ($i = 0; $i < count($_FILES['images']['name']); $i++) {
                $imageName = $_FILES['images']['name'][$i];
                $targetFile = $targetDir . basename($imageName);
                $imageFileType = strtolower(pathinfo($targetFile, PATHINFO_EXTENSION));
                if (in_array($imageFileType, $allowedTypes) && move_uploaded_file($_FILES['images']['tmp_name'][$i], $targetFile)) {
                    $additionalImages[] = '/dashboard/admin/uploads/' . $imageName;
                }
            }
            if (!empty($additionalImages) && isset($result['product_id'])) {
                $productId = $result['product_id'];
                foreach ($additionalImages as $imageUrl) {
                    $stmt = $db->prepare("INSERT INTO product_images (product_id, image_url) VALUES (:product_id, :image_url)");
                    $stmt->execute([':product_id' => $productId, ':image_url' => $imageUrl]);
                }
            }
        }
        break;

    case 'PUT':
        if (!$id) {
            http_response_code(400);
            echo json_encode(['error' => 'Product ID required']);
            exit;
        }
        $data = $_POST;
        if (!isset($data['name'], $data['description'], $data['price'], $data['stock'], $data['category_id'])) {
            http_response_code(400);
            echo json_encode(['error' => 'Missing required fields']);
            exit;
        }
        $result = $product->updateProduct($id, $data, $userId, $isVendor, $isAdmin);
        http_response_code($result['code'] ?? 200);
        echo json_encode($result);

        // Handle image updates if new files are uploaded
        if (isset($_FILES['primary_image']) && $_FILES['primary_image']['name']) {
            $primaryImage = $_FILES['primary_image']['name'];
            $targetFile = $targetDir . basename($primaryImage);
            $imageFileType = strtolower(pathinfo($targetFile, PATHINFO_EXTENSION));
            if (in_array($imageFileType, $allowedTypes) && move_uploaded_file($_FILES['primary_image']['tmp_name'], $targetFile)) {
                $primaryImage = '/dashboard/admin/uploads/' . $primaryImage;
                $stmt = $db->prepare("UPDATE products SET primary_image = :primary_image WHERE product_id = :id");
                $stmt->execute([':primary_image' => $primaryImage, ':id' => $id]);
            }
        }
        if (isset($_FILES['images']) && is_array($_FILES['images']['name'])) {
            $stmt = $db->prepare("DELETE FROM product_images WHERE product_id = :id");
            $stmt->execute([':id' => $id]);
            for ($i = 0; $i < count($_FILES['images']['name']); $i++) {
                $imageName = $_FILES['images']['name'][$i];
                $targetFile = $targetDir . basename($imageName);
                $imageFileType = strtolower(pathinfo($targetFile, PATHINFO_EXTENSION));
                if (in_array($imageFileType, $allowedTypes) && move_uploaded_file($_FILES['images']['tmp_name'][$i], $targetFile)) {
                    $imageUrl = '/dashboard/admin/uploads/' . $imageName;
                    $stmt = $db->prepare("INSERT INTO product_images (product_id, image_url) VALUES (:product_id, :image_url)");
                    $stmt->execute([':product_id' => $id, ':image_url' => $imageUrl]);
                }
            }
        }
        break;

    case 'DELETE':
        if (!$id) {
            http_response_code(400);
            echo json_encode(['error' => 'Product ID required']);
            exit;
        }
        $result = $product->deleteProduct($id, $userId, $isVendor, $isAdmin);
        http_response_code($result['code'] ?? 200);
        echo json_encode($result);
        break;

    default:
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed']);
        break;
}
?>