### API Documentation

**Base URL**: `/api/v1/`  
**Authentication**: All endpoints (except `POST /users` for registration) require an `Authorization` header with a token obtained from `/auth/login.php`.  
**Content-Type**: `application/json` for all requests and responses.

---

#### General Notes
- **Token**: A 64-character string generated upon login or registration, valid for 1 hour.
- **Roles**:
  - **Users**: Can register, review products, manage wishlist/cart/orders.
  - **Vendors**: Can manage their own products (create/update/delete), approved vendors only.
  - **Admins**: Full control over all products and can view all orders.
- **Error Responses**: Returned as `{"error": "message"}` with appropriate HTTP status codes.

---

### 1. Authentication Endpoint
#### `POST /auth/login.php`
- **Description**: Authenticate a user and return a token.
- **Request Body**:
  ```json
  {
    "email": "string",    // Required
    "password": "string"  // Required
  }
  ```
- **Responses**:
  - **200 OK**: `{"token": "random64charstring"}`
  - **400 Bad Request**: `{"error": "Email and password required"}`
  - **401 Unauthorized**: `{"error": "Invalid credentials"}`
  - **405 Method Not Allowed**: `{"error": "Method not allowed"}`

---

### 2. User Endpoints
#### `POST /api/v1/users`
- **Description**: Register a new user and return a token.
- **Request Body**:
  ```json
  {
    "email": "string",       // Required, unique
    "password": "string",    // Required
    "first_name": "string",  // Required
    "last_name": "string",   // Required
    "phone": "string"        // Optional
  }
  ```
- **Responses**:
  - **201 Created**: `{"token": "random64charstring"}`
  - **400 Bad Request**: `{"error": "Missing required fields"}` or `{"error": "Registration failed"}`
  - **405 Method Not Allowed**: `{"error": "Method not allowed"}`

---

### 3. Product Endpoints
#### `GET /api/v1/products`
- **Description**: Retrieve all active products (vendors see only their own).
- **Headers**: `Authorization: token`
- **Responses**:
  - **200 OK**: Array of product objects:
    ```json
    [
      {
        "product_id": int,
        "vendor_id": int|null,
        "name": "string",
        "description": "string",
        "price": float,
        "stock": int,
        "primary_image": "string",
        "category_id": int,
        "average_rating": float,
        "review_count": int,
        "created_at": "datetime",
        "updated_at": "datetime|null",
        "is_active": 1,
        "category_name": "string",
        "vendor_name": "string|null"
      }
    ]
    ```
  - **401 Unauthorized**: `{"error": "Unauthorized"}`

#### `GET /api/v1/products/{id}`
- **Description**: Retrieve a single active product by ID.
- **Headers**: `Authorization: token`
- **Responses**:
  - **200 OK**: Product object (same as above)
  - **404 Not Found**: `{"error": "Product not found"}`
  - **401 Unauthorized**: `{"error": "Unauthorized"}`

#### `POST /api/v1/products`
- **Description**: Create a new product (vendors tie it to themselves; admins can specify vendor_id).
- **Headers**: `Authorization: token`
- **Request Body**:
  ```json
  {
    "name": "string",         // Required
    "description": "string",  // Required
    "price": float,          // Required
    "stock": int,            // Required
    "primary_image": "string",// Required
    "category_id": int,      // Required
    "vendor_id": int         // Optional (admin only)
  }
  ```
- **Responses**:
  - **201 Created**: `{"product_id": int, "message": "Product created"}`
  - **400 Bad Request**: `{"error": "Missing required fields"}`
  - **403 Forbidden**: `{"error": "Unauthorized"}` (non-vendor/non-admin)
  - **500 Internal Server Error**: `{"error": "Failed to create product"}`
  - **401 Unauthorized**: `{"error": "Unauthorized"}`

#### `PUT /api/v1/products/{id}`
- **Description**: Update an existing product (vendors can only update their own).
- **Headers**: `Authorization: token`
- **Request Body**: Partial updates allowed
  ```json
  {
    "name": "string",
    "description": "string",
    "price": float,
    "stock": int,
    "primary_image": "string",
    "category_id": int
  }
  ```
- **Responses**:
  - **200 OK**: `{"message": "Product updated"}`
  - **400 Bad Request**: `{"error": "Product ID required"}` or `{"error": "Invalid data"}`
  - **403 Forbidden**: `{"error": "Unauthorized"}`
  - **404 Not Found**: `{"error": "Product not found"}`
  - **500 Internal Server Error**: `{"error": "Failed to update product"}`
  - **401 Unauthorized**: `{"error": "Unauthorized"}`

#### `DELETE /api/v1/products/{id}`
- **Description**: Soft delete a product (set `is_active = 0`; vendors can only delete their own).
- **Headers**: `Authorization: token`
- **Responses**:
  - **200 OK**: `{"message": "Product deleted"}`
  - **400 Bad Request**: `{"error": "Product ID required"}`
  - **403 Forbidden**: `{"error": "Unauthorized"}`
  - **404 Not Found**: `{"error": "Product not found"}`
  - **500 Internal Server Error**: `{"error": "Failed to delete product"}`
  - **401 Unauthorized**: `{"error": "Unauthorized"}`

---

### 4. Review Endpoints
#### `GET /api/v1/reviews/{product_id}`
- **Description**: Retrieve all reviews for a product.
- **Headers**: `Authorization: token`
- **Responses**:
  - **200 OK**: Array of review objects:
    ```json
    [
      {
        "review_id": int,
        "product_id": int,
        "user_id": int,
        "rating": int (1-5),
        "review_text": "string|null",
        "created_at": "datetime",
        "updated_at": "datetime|null",
        "first_name": "string",
        "last_name": "string"
      }
    ]
    ```
  - **400 Bad Request**: `{"error": "Product ID required"}`
  - **401 Unauthorized**: `{"error": "Unauthorized"}`

#### `POST /api/v1/reviews`
- **Description**: Add a review for a product.
- **Headers**: `Authorization: token`
- **Request Body**:
  ```json
  {
    "product_id": int,     // Required
    "rating": int,        // Required (1-5)
    "review_text": "string" // Optional
  }
  ```
- **Responses**:
  - **201 Created**: `{"message": "Review added"}`
  - **400 Bad Request**: `{"error": "Missing required fields"}`
  - **500 Internal Server Error**: `{"error": "Failed to add review"}`
  - **401 Unauthorized**: `{"error": "Unauthorized"}`

#### `DELETE /api/v1/reviews/{review_id}`
- **Description**: Delete a review (users can delete their own; admins can delete any).
- **Headers**: `Authorization: token`
- **Responses**:
  - **200 OK**: `{"message": "Review deleted"}`
  - **400 Bad Request**: `{"error": "Review ID required"}`
  - **404 Not Found**: `{"error": "Review not found or unauthorized"}`
  - **500 Internal Server Error**: `{"error": "Failed to delete review"}`
  - **401 Unauthorized**: `{"error": "Unauthorized"}`

---

### 5. Wishlist Endpoints
#### `GET /api/v1/wishlist`
- **Description**: Retrieve the user’s wishlist.
- **Headers**: `Authorization: token`
- **Responses**:
  - **200 OK**: Array of wishlist items:
    ```json
    [
      {
        "wishlist_id": int,
        "user_id": int,
        "product_id": int,
        "added_at": "datetime",
        "name": "string",
        "price": float,
        "primary_image": "string"
      }
    ]
    ```
  - **401 Unauthorized**: `{"error": "Unauthorized"}`

#### `POST /api/v1/wishlist`
- **Description**: Add a product to the user’s wishlist.
- **Headers**: `Authorization: token`
- **Request Body**:
  ```json
  {
    "product_id": int  // Required
  }
  ```
- **Responses**:
  - **201 Created**: `{"message": "Added to wishlist"}`
  - **400 Bad Request**: `{"error": "Product ID required"}`
  - **500 Internal Server Error**: `{"error": "Failed to add"}`
  - **401 Unauthorized**: `{"error": "Unauthorized"}`

#### `DELETE /api/v1/wishlist/{wishlist_id}`
- **Description**: Remove a product from the user’s wishlist.
- **Headers**: `Authorization: token`
- **Responses**:
  - **200 OK**: `{"message": "Removed from wishlist"}`
  - **400 Bad Request**: `{"error": "Wishlist ID required"}`
  - **500 Internal Server Error**: `{"error": "Failed to remove"}`
  - **401 Unauthorized**: `{"error": "Unauthorized"}`

---

### 6. Cart Endpoints
#### `GET /api/v1/cart`
- **Description**: Retrieve the user’s cart.
- **Headers**: `Authorization: token`
- **Responses**:
  - **200 OK**: Array of cart items:
    ```json
    [
      {
        "cart_id": int,
        "user_id": int,
        "product_id": int,
        "quantity": int,
        "added_at": "datetime",
        "name": "string",
        "price": float,
        "primary_image": "string"
      }
    ]
    ```
  - **401 Unauthorized**: `{"error": "Unauthorized"}`

#### `POST /api/v1/cart`
- **Description**: Add a product to the user’s cart (increments quantity if already present).
- **Headers**: `Authorization: token`
- **Request Body**:
  ```json
  {
    "product_id": int,  // Required
    "quantity": int    // Required, > 0
  }
  ```
- **Responses**:
  - **201 Created**: `{"message": "Added to cart"}`
  - **400 Bad Request**: `{"error": "Product ID and quantity required"}`
  - **500 Internal Server Error**: `{"error": "Failed to add"}`
  - **401 Unauthorized**: `{"error": "Unauthorized"}`

#### `PUT /api/v1/cart/{cart_id}`
- **Description**: Update the quantity of an item in the user’s cart.
- **Headers**: `Authorization: token`
- **Request Body**:
  ```json
  {
    "quantity": int  // Required, > 0
  }
  ```
- **Responses**:
  - **200 OK**: `{"message": "Cart updated"}`
  - **400 Bad Request**: `{"error": "Cart ID required"}` or `{"error": "Quantity required"}`
  - **500 Internal Server Error**: `{"error": "Failed to update"}`
  - **401 Unauthorized**: `{"error": "Unauthorized"}`

#### `DELETE /api/v1/cart/{cart_id}`
- **Description**: Remove an item from the user’s cart.
- **Headers**: `Authorization: token`
- **Responses**:
  - **200 OK**: `{"message": "Removed from cart"}`
  - **400 Bad Request**: `{"error": "Cart ID required"}`
  - **500 Internal Server Error**: `{"error": "Failed to remove"}`
  - **401 Unauthorized**: `{"error": "Unauthorized"}`

---

### 7. Order Endpoints
#### `GET /api/v1/orders`
- **Description**: Retrieve all orders (users see only their own; admins see all).
- **Headers**: `Authorization: token`
- **Responses**:
  - **200 OK**: Array of order objects:
    ```json
    [
      {
        "order_id": int,
        "user_id": int,
        "total_amount": float,
        "status": "string",
        "shipping_address": "string",
        "billing_address": "string|null",
        "payment_method": "string",
        "payment_status": "string",
        "created_at": "datetime",
        "updated_at": "datetime|null"
      }
    ]
    ```
  - **401 Unauthorized**: `{"error": "Unauthorized"}`

#### `GET /api/v1/orders/{order_id}`
- **Description**: Retrieve a single order (users see only their own; admins see any).
- **Headers**: `Authorization: token`
- **Responses**:
  - **200 OK**: Order object (same as above)
  - **404 Not Found**: `{"error": "Order not found"}`
  - **401 Unauthorized**: `{"error": "Unauthorized"}`

#### `POST /api/v1/orders`
- **Description**: Create an order from the user’s cart, clear cart, and update sales.
- **Headers**: `Authorization: token`
- **Request Body**:
  ```json
  {
    "shipping_address": "string",  // Required
    "payment_method": "string"    // Required (e.g., "Credit Card", "PayPal")
  }
  ```
- **Responses**:
  - **201 Created**: `{"order_id": int, "message": "Order created"}`
  - **400 Bad Request**: `{"error": "Shipping address and payment method required"}`
  - **500 Internal Server Error**: `{"error": "Failed to create order: [message]"}`
  - **401 Unauthorized**: `{"error": "Unauthorized"}`

---

### Summary of Routes
| Method | Endpoint                  | Description                          | Authentication Required |
|--------|---------------------------|--------------------------------------|-------------------------|
| POST   | `/auth/login.php`         | User login                           | No                      |
| POST   | `/api/v1/users`           | User registration                    | No                      |
| GET    | `/api/v1/products`        | List all products                    | Yes                     |
| GET    | `/api/v1/products/{id}`   | Get a product                        | Yes                     |
| POST   | `/api/v1/products`        | Create a product                     | Yes (vendor/admin)      |
| PUT    | `/api/v1/products/{id}`   | Update a product                     | Yes (vendor/admin)      |
| DELETE | `/api/v1/products/{id}`   | Delete a product                     | Yes (vendor/admin)      |
| GET    | `/api/v1/reviews/{product_id}` | Get reviews for a product       | Yes                     |
| POST   | `/api/v1/reviews`         | Add a review                         | Yes                     |
| DELETE | `/api/v1/reviews/{review_id}` | Delete a review                  | Yes                     |
| GET    | `/api/v1/wishlist`        | Get user’s wishlist                  | Yes                     |
| POST   | `/api/v1/wishlist`        | Add to wishlist                      | Yes                     |
| DELETE | `/api/v1/wishlist/{wishlist_id}` | Remove from wishlist          | Yes                     |
| GET    | `/api/v1/cart`            | Get user’s cart                      | Yes                     |
| POST   | `/api/v1/cart`            | Add to cart                          | Yes                     |
| PUT    | `/api/v1/cart/{cart_id}`  | Update cart item quantity            | Yes                     |
| DELETE | `/api/v1/cart/{cart_id}`  | Remove from cart                     | Yes                     |
| GET    | `/api/v1/orders`          | List orders                          | Yes                     |
| GET    | `/api/v1/orders/{order_id}` | Get an order                       | Yes                     |
| POST   | `/api/v1/orders`          | Create an order                      | Yes                     |

---




insert initial admin walkthrough
If you want to hash admin passwords (which is a best practice for security), you’ll need to insert the initial admin into the `admins` table with a hashed password. Since the password is hashed (e.g., using `password_hash()` with the `PASSWORD_BCRYPT` algorithm), you cannot insert it as plain text directly into the database in the same way you would with a plain text password. Instead, you’ll need to generate the hashed password programmatically or manually and then insert it.

Here are a few methods to insert the initial admin with a hashed password:

---

### Method 1: Insert via PHP Script
You can create a one-time PHP script to insert the initial admin with a hashed password. This script can be run once to set up the admin account.

#### Example Script (`insert_initial_admin.php`)
```php
<?php
require_once 'backend/classes/Database.php';

try {
    $db = (new Database())->getConnection();

    // Admin details
    $email = 'admin@example.com';
    $password = 'admin123'; // Plain text password to hash
    $firstName = 'Admin';
    $lastName = 'User';

    // Hash the password
    $hashedPassword = password_hash($password, PASSWORD_BCRYPT);

    // Prepare and execute the insert statement
    $stmt = $db->prepare(
        "INSERT INTO admins (email, password, first_name, last_name, created_at) 
         VALUES (:email, :password, :first_name, :last_name, NOW())"
    );
    $stmt->execute([
        'email' => $email,
        'password' => $hashedPassword,
        'first_name' => $firstName,
        'last_name' => $lastName
    ]);

    echo "Initial admin inserted successfully with email: $email and hashed password.";
} catch (PDOException $e) {
    echo "Error: " . $e->getMessage();
}
?>
```

#### Steps
1. **Place the Script**: Save this as `/dashboard/admin/insert_initial_admin.php` (or any location in your project).
2. **Run the Script**: Execute it from the command line:
   ```bash
   php /home/cyberpunkmania/Documents/Github/roamingndovu/RoamingNdovu/dashboard/admin/insert_initial_admin.php
   ```
3. **Verify**: Check the `admins` table in your database to confirm the insertion.
4. **Delete the Script**: After running, delete `insert_initial_admin.php` or move it outside the web root to prevent reuse.

#### Notes
- Replace `'admin@example.com'`, `'admin123'`, `'Admin'`, and `'User'` with your desired values.
- The `created_at` column is set to `NOW()` automatically.

---

### Method 2: Insert via SQL with Manual Hashing
If you prefer to do this directly in the database or via a SQL client (e.g., phpMyAdmin, MySQL Workbench), you can hash the password manually using PHP or a tool, then insert it.

#### Steps
1. **Hash the Password**:
   - Use a PHP one-liner to generate the hash:
     ```php
     echo password_hash('admin123', PASSWORD_BCRYPT);
     ```
   - Example output: `$2y$10$randomstring...` (a 60-character hash).
   - Run this in a PHP file or terminal and copy the output.

2. **Insert into Database**:
   - Use a SQL command:
     ```sql
     INSERT INTO admins (email, password, first_name, last_name, created_at)
     VALUES ('admin@example.com', '$2y$10$randomstring...', 'Admin', 'User', NOW());
     ```
   - Replace `$2y$10$randomstring...` with the actual hash you generated.

3. **Verify**: Check the `admins` table to ensure the record is added.

#### Notes
- This method requires manual hashing and is less automated but works if you don’t want to write a script.
- Ensure the hash matches the `VARCHAR(255)` length in the `admins` table schema.

---

### Method 3: Update Existing Admin via PHP
If you already have an `admins` table but the password is plain text, you can update it with a hashed version.

#### Example Script (`update_admin_password.php`)
```php
<?php
require_once 'backend/classes/Database.php';

try {
    $db = (new Database())->getConnection();

    // Admin email to update
    $email = 'admin@example.com';
    $newPassword = 'admin123'; // New plain text password to hash

    // Hash the new password
    $hashedPassword = password_hash($newPassword, PASSWORD_BCRYPT);

    // Update the password
    $stmt = $db->prepare(
        "UPDATE admins SET password = :password WHERE email = :email"
    );
    $stmt->execute([
        'password' => $hashedPassword,
        'email' => $email
    ]);

    echo "Admin password updated successfully for email: $email.";
} catch (PDOException $e) {
    echo "Error: " . $e->getMessage();
}
?>
```

#### Steps
1. **Place the Script**: Save as `/dashboard/admin/update_admin_password.php`.
2. **Run the Script**:
   ```bash
   php /home/cyberpunkmania/Documents/Github/roamingndovu/RoamingNdovu/dashboard/admin/update_admin_password.php
   ```
3. **Verify**: Check the `admins` table to confirm the `password` column now contains a hash.
4. **Delete the Script**: Remove it after use.

---

### Integration with Login
Since your `/dashboard/admin/login.php` already uses `password_verify()` to check the hashed password, no changes are needed there. The login process will work seamlessly with the hashed password once it’s inserted.

#### Example Login with Hashed Password
- Inserted admin: `email = 'admin@example.com'`, `password = $2y$10$randomstring...`
- Login attempt with `email = 'admin@example.com'` and `password = 'admin123'` will succeed because `password_verify('admin123', $2y$10$randomstring...)` returns `true`.

---

### Best Practices
- **Security**: Always hash passwords with `password_hash()` and verify with `password_verify()`. Never store plain text passwords.
- **Initial Setup**: Use Method 1 or 2 for the first admin, then rely on the login system for future admins.
- **Backup**: Before running any script, back up your database to avoid accidental data loss.

---

### Final Recommendation
Use **Method 1 (PHP Script)** for the initial admin setup. It’s automated, secure, and aligns with your existing backend structure. After running it, you can log in with the credentials and proceed with your admin panel.
