<?php
require_once __DIR__ . '/Database.php';

class Order {
    private $db;

    public function __construct() {
        $this->db = (new Database())->getConnection();
    }

    public function getOrders($userId = null, $isAdmin = false) {
        $query = "SELECT o.*, (SELECT SUM(oi.quantity * oi.price) FROM order_items oi WHERE oi.order_id = o.order_id) AS total_amount 
                  FROM orders o";
        $params = [];
        if (!$isAdmin && $userId !== null) {
            $query .= " WHERE o.user_id = :user_id";
            $params[':user_id'] = $userId;
        }
        $stmt = $this->db->prepare($query);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getOrder($orderId, $userId, $isAdmin) {
        $query = "SELECT o.*, (SELECT SUM(oi.quantity * oi.price) FROM order_items oi WHERE oi.order_id = o.order_id) AS total_amount 
                  FROM orders o WHERE o.order_id = :order_id";
        $params = ['order_id' => $orderId];
        if (!$isAdmin) {
            $query .= " AND o.user_id = :user_id";
            $params['user_id'] = $userId;
        }
        $stmt = $this->db->prepare($query);
        $stmt->execute($params);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function createOrder($userId, $shippingAddress, $paymentMethod) {
        $this->db->beginTransaction();
        try {
            // Insert order
            $stmt = $this->db->prepare(
                "INSERT INTO orders (user_id, shipping_address, payment_method, created_at) 
                 VALUES (:user_id, :shipping_address, :payment_method, NOW())"
            );
            $stmt->execute([
                'user_id' => $userId,
                'shipping_address' => $shippingAddress,
                'payment_method' => $paymentMethod
            ]);
            $orderId = $this->db->lastInsertId();

            // Move cart items to order_items and update product stock
            $stmt = $this->db->prepare(
                "INSERT INTO order_items (order_id, product_id, quantity, price) 
                 SELECT :order_id, c.product_id, c.quantity, p.price 
                 FROM cart c 
                 JOIN products p ON c.product_id = p.product_id 
                 WHERE c.user_id = :user_id"
            );
            $stmt->execute(['order_id' => $orderId, 'user_id' => $userId]);

            // Update stock in products table
            $stmt = $this->db->prepare(
                "UPDATE products p 
                 JOIN order_items oi ON p.product_id = oi.product_id 
                 SET p.stock = p.stock - oi.quantity 
                 WHERE oi.order_id = :order_id"
            );
            $stmt->execute(['order_id' => $orderId]);

            // Clear cart
            $stmt = $this->db->prepare("DELETE FROM cart WHERE user_id = :user_id");
            $stmt->execute(['user_id' => $userId]);

            // Update sales (for analytics)
            $stmt = $this->db->prepare(
                "INSERT INTO sales (order_id, vendor_id, product_id, quantity, total) 
                 SELECT :order_id, p.vendor_id, p.product_id, oi.quantity, (oi.quantity * oi.price) 
                 FROM order_items oi 
                 JOIN products p ON oi.product_id = p.product_id 
                 WHERE oi.order_id = :order_id"
            );
            $stmt->execute(['order_id' => $orderId]);

            $this->db->commit();
            return ['order_id' => $orderId, 'message' => 'Order created successfully', 'code' => 201];
        } catch (Exception $e) {
            $this->db->rollBack();
            error_log("Order creation failed: " . $e->getMessage());
            return ['error' => 'Failed to create order: ' . $e->getMessage(), 'code' => 500];
        }
    }

    public function deleteOrder($orderId, $userId, $isAdmin) {
        $this->db->beginTransaction();
        try {
            $order = $this->getOrder($orderId, $userId, $isAdmin);
            if (!$order) {
                throw new Exception('Order not found or unauthorized');
            }

            // Restore stock for cancelled order
            $stmt = $this->db->prepare(
                "UPDATE products p 
                 JOIN order_items oi ON p.product_id = oi.product_id 
                 SET p.stock = p.stock + oi.quantity 
                 WHERE oi.order_id = :order_id"
            );
            $stmt->execute(['order_id' => $orderId]);

            // Delete order items
            $stmt = $this->db->prepare("DELETE FROM order_items WHERE order_id = :order_id");
            $stmt->execute(['order_id' => $orderId]);

            // Delete sales records
            $stmt = $this->db->prepare("DELETE FROM sales WHERE order_id = :order_id");
            $stmt->execute(['order_id' => $orderId]);

            // Delete order
            $stmt = $this->db->prepare("DELETE FROM orders WHERE order_id = :order_id");
            $stmt->execute(['order_id' => $orderId]);

            $this->db->commit();
            return ['message' => 'Order cancelled successfully', 'code' => 200];
        } catch (Exception $e) {
            $this->db->rollBack();
            error_log("Order deletion failed: " . $e->getMessage());
            return ['error' => 'Failed to cancel order: ' . $e->getMessage(), 'code' => 500];
        }
    }
}
?>