<?php
require_once __DIR__ . '/Database.php';
require_once __DIR__ . '/Vendor.php';

class Product {
    private $db;
    private $vendor;

    public function __construct() {
        $this->db = (new Database())->getConnection();
        $this->vendor = new Vendor();
    }

    // Get a single product by ID
    public function getProduct($productId) {
        $stmt = $this->db->prepare(
            "SELECT p.*, c.name AS category_name, v.vendor_name 
             FROM products p 
             LEFT JOIN categories c ON p.category_id = c.category_id 
             LEFT JOIN vendors v ON p.vendor_id = v.vendor_id 
             WHERE p.product_id = :id AND p.is_active = 1"
        );
        $stmt->execute(['id' => $productId]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    // Get all products (optionally filtered by vendor for vendors)
    public function getAllProducts($userId, $isVendor, $isAdmin) {
        $query = "SELECT p.*, c.name AS category_name, v.vendor_name 
                  FROM products p 
                  LEFT JOIN categories c ON p.category_id = c.category_id 
                  LEFT JOIN vendors v ON p.vendor_id = v.vendor_id 
                  WHERE p.is_active = 1";
        if ($isVendor && !$isAdmin) {
            $query .= " AND p.vendor_id = :user_id";
        }
        $stmt = $this->db->prepare($query);
        if ($isVendor && !$isAdmin) {
            $stmt->execute(['user_id' => $userId]);
        } else {
            $stmt->execute();
        }
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // Create a new product
    public function createProduct($data, $userId, $isVendor, $isAdmin) {
        if (!$isVendor && !$isAdmin) {
            return ['error' => 'Unauthorized', 'code' => 403];
        }

        $vendorId = $isVendor && !$isAdmin ? $userId : ($data['vendor_id'] ?? null);
        $stmt = $this->db->prepare(
            "INSERT INTO products (vendor_id, name, description, price, stock, primary_image, category_id) 
             VALUES (:vendor_id, :name, :description, :price, :stock, :primary_image, :category_id)"
        );
        $result = $stmt->execute([
            'vendor_id' => $vendorId,
            'name' => $data['name'],
            'description' => $data['description'],
            'price' => $data['price'],
            'stock' => $data['stock'],
            'primary_image' => $data['primary_image'],
            'category_id' => $data['category_id']
        ]);

        if ($result) {
            return ['product_id' => $this->db->lastInsertId(), 'message' => 'Product created'];
        }
        return ['error' => 'Failed to create product', 'code' => 500];
    }

    // Update an existing product
    public function updateProduct($productId, $data, $userId, $isVendor, $isAdmin) {
        $product = $this->getProduct($productId);
        if (!$product) {
            return ['error' => 'Product not found', 'code' => 404];
        }

        $isOwner = $product['vendor_id'] == $userId;
        if (!$isAdmin && ($isVendor && !$isOwner)) {
            return ['error' => 'Unauthorized', 'code' => 403];
        }

        $stmt = $this->db->prepare(
            "UPDATE products 
             SET name = :name, description = :description, price = :price, stock = :stock, 
                 primary_image = :primary_image, category_id = :category_id 
             WHERE product_id = :id"
        );
        $result = $stmt->execute([
            'id' => $productId,
            'name' => $data['name'] ?? $product['name'],
            'description' => $data['description'] ?? $product['description'],
            'price' => $data['price'] ?? $product['price'],
            'stock' => $data['stock'] ?? $product['stock'],
            'primary_image' => $data['primary_image'] ?? $product['primary_image'],
            'category_id' => $data['category_id'] ?? $product['category_id']
        ]);

        return $result ? ['message' => 'Product updated'] : ['error' => 'Failed to update product', 'code' => 500];
    }

    // Delete a product (soft delete by setting is_active = 0)
    public function deleteProduct($productId, $userId, $isVendor, $isAdmin) {
        $product = $this->getProduct($productId);
        if (!$product) {
            return ['error' => 'Product not found', 'code' => 404];
        }

        $isOwner = $product['vendor_id'] == $userId;
        if (!$isAdmin && ($isVendor && !$isOwner)) {
            return ['error' => 'Unauthorized', 'code' => 403];
        }

        $stmt = $this->db->prepare(
            "UPDATE products SET is_active = 0 WHERE product_id = :id"
        );
        $result = $stmt->execute(['id' => $productId]);

        return $result ? ['message' => 'Product deleted'] : ['error' => 'Failed to delete product', 'code' => 500];
    }
}
?>