<?php
require_once __DIR__ . '/Database.php';

class Review {
    private $db;

    public function __construct() {
        $this->db = (new Database())->getConnection();
    }

    public function getReviews($productId) {
        $stmt = $this->db->prepare(
            "SELECT r.*, u.first_name, u.last_name 
             FROM product_reviews r 
             JOIN users u ON r.user_id = u.user_id 
             WHERE r.product_id = :product_id"
        );
        $stmt->execute(['product_id' => $productId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function createReview($productId, $userId, $rating, $reviewText) {
        $stmt = $this->db->prepare(
            "INSERT INTO product_reviews (product_id, user_id, rating, review_text) 
             VALUES (:product_id, :user_id, :rating, :review_text)"
        );
        $result = $stmt->execute([
            'product_id' => $productId,
            'user_id' => $userId,
            'rating' => $rating,
            'review_text' => $reviewText
        ]);

        if ($result) {
            $this->updateProductRating($productId);
            return ['message' => 'Review added'];
        }
        return ['error' => 'Failed to add review', 'code' => 500];
    }

    public function deleteReview($reviewId, $userId, $isAdmin) {
        $stmt = $this->db->prepare(
            "SELECT product_id, user_id FROM product_reviews WHERE review_id = :review_id"
        );
        $stmt->execute(['review_id' => $reviewId]);
        $review = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$review || (!$isAdmin && $review['user_id'] != $userId)) {
            return ['error' => 'Review not found or unauthorized', 'code' => 404];
        }

        $stmt = $this->db->prepare("DELETE FROM product_reviews WHERE review_id = :review_id");
        $result = $stmt->execute(['review_id' => $reviewId]);

        if ($result) {
            $this->updateProductRating($review['product_id']);
            return ['message' => 'Review deleted'];
        }
        return ['error' => 'Failed to delete review', 'code' => 500];
    }

    private function updateProductRating($productId) {
        $stmt = $this->db->prepare(
            "UPDATE products 
             SET average_rating = (SELECT AVG(rating) FROM product_reviews WHERE product_id = :product_id), 
                 review_count = (SELECT COUNT(*) FROM product_reviews WHERE product_id = :product_id) 
             WHERE product_id = :product_id"
        );
        $stmt->execute(['product_id' => $productId]);
    }
}
?>