<?php
require_once __DIR__ . '/Database.php';
require_once __DIR__ . '/Auth.php';

class User {
    private $db;

    public function __construct() {
        $this->db = (new Database())->getConnection();
    }

    public function register($email, $password, $firstName, $lastName, $phone = null, $role = 'user') {
        $stmt = $this->db->prepare("INSERT INTO users (email, password, first_name, last_name, phone, role, created_at) VALUES (:email, :password, :first_name, :last_name, :phone, :role, NOW())");
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
        $result = $stmt->execute([
            ':email' => $email,
            ':password' => $hashedPassword,
            ':first_name' => $firstName,
            ':last_name' => $lastName,
            ':phone' => $phone,
            ':role' => $role
        ]);

        if ($result) {
            $userId = $this->db->lastInsertId();
            $auth = new Auth();
            return $auth->generateToken($userId);
        }
        return false;
    }

    public function login($email, $password) {
        $stmt = $this->db->prepare("SELECT user_id, password FROM users WHERE email = :email");
        $stmt->execute([':email' => $email]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($user && password_verify($password, $user['password'])) {
            $auth = new Auth();
            return $auth->generateToken($user['user_id']);
        }
        return false;
    }

    public function getAllUsers() {
        $stmt = $this->db->query("SELECT user_id, email, first_name, last_name, phone, created_at, role FROM users");
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function getUser($userId) {
        $stmt = $this->db->prepare("SELECT user_id, email, first_name, last_name, phone, created_at, role FROM users WHERE user_id = :user_id");
        $stmt->execute([':user_id' => $userId]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    public function updateUser($userId, $data) {
        $fields = [];
        $params = [':user_id' => $userId];
        if (isset($data['email'])) $fields[] = "email = :email"; $params[':email'] = $data['email'];
        if (isset($data['password'])) {
            $fields[] = "password = :password";
            $params[':password'] = password_hash($data['password'], PASSWORD_DEFAULT);
        }
        if (isset($data['first_name'])) $fields[] = "first_name = :first_name"; $params[':first_name'] = $data['first_name'];
        if (isset($data['last_name'])) $fields[] = "last_name = :last_name"; $params[':last_name'] = $data['last_name'];
        if (isset($data['phone'])) $fields[] = "phone = :phone"; $params[':phone'] = $data['phone'];
        if (isset($data['role'])) $fields[] = "role = :role"; $params[':role'] = $data['role'];

        if (empty($fields)) return false;

        $stmt = $this->db->prepare("UPDATE users SET " . implode(', ', $fields) . " WHERE user_id = :user_id");
        return $stmt->execute($params);
    }

    public function deleteUser($userId) {
        $stmt = $this->db->prepare("DELETE FROM users WHERE user_id = :user_id");
        return $stmt->execute([':user_id' => $userId]);
    }
}
?>
