### Updated Database Schema for E-commerce Site

#### 1. `users` Table
Stores customer information with simple registration.
- **Purpose**: Manage customer accounts for shopping, reviews, and wishlists.
- **Columns**:
  - `user_id` (INT, PRIMARY KEY, AUTO_INCREMENT) - Unique identifier for each user.
  - `email` (VARCHAR(255), UNIQUE, NOT NULL) - User’s email for login.
  - `password` (VARCHAR(255), NOT NULL) - Hashed password for security.
  - `first_name` (VARCHAR(100), NOT NULL) - User’s first name.
  - `last_name` (VARCHAR(100), NOT NULL) - User’s last name.
  - `phone` (VARCHAR(20), NULL) - Optional phone number.
  - `created_at` (DATETIME, DEFAULT CURRENT_TIMESTAMP) - Timestamp of account creation.
  - `updated_at` (DATETIME, NULL) - Timestamp of last update.

#### 2. `admins` Table
Stores admin accounts, separate from users for role-based access.
- **Purpose**: Manage admins who oversee the platform and approve vendors.
- **Columns**:
  - `admin_id` (INT, PRIMARY KEY, AUTO_INCREMENT) - Unique identifier for each admin.
  - `email` (VARCHAR(255), UNIQUE, NOT NULL) - Admin’s email for login.
  - `password` (VARCHAR(255), NOT NULL) - Hashed password.
  - `first_name` (VARCHAR(100), NOT NULL) - Admin’s first name.
  - `last_name` (VARCHAR(100), NOT NULL) - Admin’s last name.
  - `created_at` (DATETIME, DEFAULT CURRENT_TIMESTAMP) - Timestamp of account creation.

#### 3. `vendors` Table
Stores vendor accounts, added and approved by admins.
- **Purpose**: Manage vendors who upload and manage products.
- **Columns**:
  - `vendor_id` (INT, PRIMARY KEY, AUTO_INCREMENT) - Unique identifier for each vendor.
  - `email` (VARCHAR(255), UNIQUE, NOT NULL) - Vendor’s email for login.
  - `password` (VARCHAR(255), NOT NULL) - Hashed password.
  - `vendor_name` (VARCHAR(100), NOT NULL) - Vendor’s business or display name.
  - `phone` (VARCHAR(20), NULL) - Optional contact number.
  - `created_by` (INT, FOREIGN KEY -> `admins.admin_id`, NOT NULL) - Admin who added the vendor.
  - `created_at` (DATETIME, DEFAULT CURRENT_TIMESTAMP) - Timestamp of account creation.
  - `approved` (TINYINT(1), DEFAULT 0) - Approval status (0 = pending, 1 = approved).
  - `bio` (TEXT, NULL) - Optional vendor description or profile info.

#### 4. `categories` Table
Stores product categories.
- **Purpose**: Organize products into categories for browsing and filtering.
- **Columns**:
  - `category_id` (INT, PRIMARY KEY, AUTO_INCREMENT) - Unique identifier for each category.
  - `name` (VARCHAR(100), UNIQUE, NOT NULL) - Category name (e.g., Electronics, Clothing).
  - `description` (TEXT, NULL) - Optional category description.
  - `created_at` (DATETIME, DEFAULT CURRENT_TIMESTAMP) - Timestamp of creation.

#### 5. `products` Table
Stores product details uploaded by admins or vendors, with support for ratings.
- **Purpose**: Core table for the product catalog.
- **Columns**:
  - `product_id` (INT, PRIMARY KEY, AUTO_INCREMENT) - Unique identifier for each product.
  - `vendor_id` (INT, FOREIGN KEY -> `vendors.vendor_id`, NULL) - Vendor who uploaded (NULL if admin-uploaded).
  - `name` (VARCHAR(255), NOT NULL) - Product name.
  - `description` (TEXT, NOT NULL) - Detailed product description.
  - `price` (DECIMAL(10,2), NOT NULL) - Current price of the product.
  - `stock` (INT, NOT NULL) - Available quantity in inventory.
  - `primary_image` (VARCHAR(255), NOT NULL) - Path or URL to the primary product image.
  - `category_id` (INT, FOREIGN KEY -> `categories.category_id`, NOT NULL) - Product category.
  - `average_rating` (DECIMAL(3,2), DEFAULT 0.00) - Average rating (e.g., 4.75), calculated from reviews.
  - `review_count` (INT, DEFAULT 0) - Number of reviews for the product.
  - `created_at` (DATETIME, DEFAULT CURRENT_TIMESTAMP) - Timestamp of product creation.
  - `updated_at` (DATETIME, NULL) - Timestamp of last update.
  - `is_active` (TINYINT(1), DEFAULT 1) - Product status (1 = active, 0 = inactive).

#### 6. `product_images` Table
Stores additional images for products (up to 5, including primary in `products` table).
- **Purpose**: Allow multiple images per product, with one marked as primary in the `products` table.
- **Columns**:
  - `image_id` (INT, PRIMARY KEY, AUTO_INCREMENT) - Unique identifier for each image.
  - `product_id` (INT, FOREIGN KEY -> `products.product_id`, NOT NULL) - Product this image belongs to.
  - `image_url` (VARCHAR(255), NOT NULL) - Path or URL to the image.
  - `created_at` (DATETIME, DEFAULT CURRENT_TIMESTAMP) - Timestamp of image upload.

#### 7. `product_reviews` Table
Stores user ratings and reviews for products.
- **Purpose**: Allow users to rate (1-5 stars) and review products.
- **Columns**:
  - `review_id` (INT, PRIMARY KEY, AUTO_INCREMENT) - Unique identifier for each review.
  - `product_id` (INT, FOREIGN KEY -> `products.product_id`, NOT NULL) - Product being reviewed.
  - `user_id` (INT, FOREIGN KEY -> `users.user_id`, NOT NULL) - User who wrote the review.
  - `rating` (TINYINT, NOT NULL, CHECK (rating BETWEEN 1 AND 5)) - Rating from 1 to 5.
  - `review_text` (TEXT, NULL) - Optional review comment.
  - `created_at` (DATETIME, DEFAULT CURRENT_TIMESTAMP) - Timestamp of review creation.
  - `updated_at` (DATETIME, NULL) - Timestamp of last update (if edited).

#### 8. `wishlist` Table
Stores products users want to save for later.
- **Purpose**: Allow users to maintain a wishlist.
- **Columns**:
  - `wishlist_id` (INT, PRIMARY KEY, AUTO_INCREMENT) - Unique identifier for each wishlist entry.
  - `user_id` (INT, FOREIGN KEY -> `users.user_id`, NOT NULL) - User who added the product.
  - `product_id` (INT, FOREIGN KEY -> `products.product_id`, NOT NULL) - Product in the wishlist.
  - `added_at` (DATETIME, DEFAULT CURRENT_TIMESTAMP) - Timestamp when added.

#### 9. `cart` Table
Stores items added to a user’s shopping cart.
- **Purpose**: Temporary storage for items before checkout.
- **Columns**:
  - `cart_id` (INT, PRIMARY KEY, AUTO_INCREMENT) - Unique identifier for each cart entry.
  - `user_id` (INT, FOREIGN KEY -> `users.user_id`, NOT NULL) - User who owns the cart.
  - `product_id` (INT, FOREIGN KEY -> `products.product_id`, NOT NULL) - Product added to cart.
  - `quantity` (INT, NOT NULL, CHECK (quantity > 0)) - Number of items.
  - `added_at` (DATETIME, DEFAULT CURRENT_TIMESTAMP) - Timestamp when added to cart.

#### 10. `orders` Table
Stores completed orders with detailed status tracking.
- **Purpose**: Track customer purchases and their lifecycle.
- **Columns**:
  - `order_id` (INT, PRIMARY KEY, AUTO_INCREMENT) - Unique identifier for each order.
  - `user_id` (INT, FOREIGN KEY -> `users.user_id`, NOT NULL) - Customer who placed the order.
  - `total_amount` (DECIMAL(10,2), NOT NULL) - Total order cost (including tax/shipping if applicable).
  - `status` (ENUM('pending', 'processing', 'shipped', 'delivered', 'cancelled'), DEFAULT 'pending') - Order status.
  - `shipping_address` (TEXT, NOT NULL) - Full delivery address.
  - `billing_address` (TEXT, NULL) - Optional separate billing address.
  - `payment_method` (VARCHAR(50), NOT NULL) - e.g., "Credit Card", "PayPal", "Cash on Delivery".
  - `payment_status` (ENUM('pending', 'paid', 'failed'), DEFAULT 'pending') - Payment status.
  - `created_at` (DATETIME, DEFAULT CURRENT_TIMESTAMP) - Timestamp of order creation.
  - `updated_at` (DATETIME, NULL) - Timestamp of last update.

#### 11. `order_items` Table
Stores individual items within an order.
- **Purpose**: Link products to orders with quantities and prices at the time of purchase.
- **Columns**:
  - `order_item_id` (INT, PRIMARY KEY, AUTO_INCREMENT) - Unique identifier for each item.
  - `order_id` (INT, FOREIGN KEY -> `orders.order_id`, NOT NULL) - Order this item belongs to.
  - `product_id` (INT, FOREIGN KEY -> `products.product_id`, NOT NULL) - Product purchased.
  - `quantity` (INT, NOT NULL, CHECK (quantity > 0)) - Number of items ordered.
  - `price` (DECIMAL(10,2), NOT NULL) - Price per unit at the time of purchase (for historical accuracy).

#### 12. `sales` Table
Tracks sales data for reporting and analytics.
- **Purpose**: Provide sales insights for admins and vendors.
- **Columns**:
  - `sale_id` (INT, PRIMARY KEY, AUTO_INCREMENT) - Unique identifier for each sale record.
  - `order_id` (INT, FOREIGN KEY -> `orders.order_id`, NOT NULL) - Related order.
  - `vendor_id` (INT, FOREIGN KEY -> `vendors.vendor_id`, NULL) - Vendor who sold the product (NULL if admin-uploaded).
  - `product_id` (INT, FOREIGN KEY -> `products.product_id`, NOT NULL) - Product sold.
  - `quantity` (INT, NOT NULL) - Number of items sold.
  - `total` (DECIMAL(10,2), NOT NULL) - Total revenue from this sale (quantity × price).
  - `sale_date` (DATETIME, DEFAULT CURRENT_TIMESTAMP) - Timestamp of the sale.

---

### SQL Code to Create the Schema
Here’s the complete SQL code to create the database and tables:

```sql
-- Create the database
CREATE DATABASE ecommerce_db;
USE ecommerce_db;

-- Users table
CREATE TABLE users (
    user_id INT PRIMARY KEY AUTO_INCREMENT,
    email VARCHAR(255) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    first_name VARCHAR(100) NOT NULL,
    last_name VARCHAR(100) NOT NULL,
    phone VARCHAR(20),
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT NULL
);

-- Admins table
CREATE TABLE admins (
    admin_id INT PRIMARY KEY AUTO_INCREMENT,
    email VARCHAR(255) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    first_name VARCHAR(100) NOT NULL,
    last_name VARCHAR(100) NOT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP
);

-- Vendors table
CREATE TABLE vendors (
    vendor_id INT PRIMARY KEY AUTO_INCREMENT,
    email VARCHAR(255) UNIQUE NOT NULL,
    password VARCHAR(255) NOT NULL,
    vendor_name VARCHAR(100) NOT NULL,
    phone VARCHAR(20),
    created_by INT NOT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    approved TINYINT(1) DEFAULT 0,
    bio TEXT,
    FOREIGN KEY (created_by) REFERENCES admins(admin_id)
);

-- Categories table
CREATE TABLE categories (
    category_id INT PRIMARY KEY AUTO_INCREMENT,
    name VARCHAR(100) UNIQUE NOT NULL,
    description TEXT,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP
);

-- Products table
CREATE TABLE products (
    product_id INT PRIMARY KEY AUTO_INCREMENT,
    vendor_id INT,
    name VARCHAR(255) NOT NULL,
    description TEXT NOT NULL,
    price DECIMAL(10,2) NOT NULL,
    stock INT NOT NULL,
    primary_image VARCHAR(255) NOT NULL,
    category_id INT NOT NULL,
    average_rating DECIMAL(3,2) DEFAULT 0.00,
    review_count INT DEFAULT 0,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT NULL,
    is_active TINYINT(1) DEFAULT 1,
    FOREIGN KEY (vendor_id) REFERENCES vendors(vendor_id),
    FOREIGN KEY (category_id) REFERENCES categories(category_id)
);

-- Product Images table
CREATE TABLE product_images (
    image_id INT PRIMARY KEY AUTO_INCREMENT,
    product_id INT NOT NULL,
    image_url VARCHAR(255) NOT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (product_id) REFERENCES products(product_id)
);

-- Product Reviews table
CREATE TABLE product_reviews (
    review_id INT PRIMARY KEY AUTO_INCREMENT,
    product_id INT NOT NULL,
    user_id INT NOT NULL,
    rating TINYINT NOT NULL CHECK (rating BETWEEN 1 AND 5),
    review_text TEXT,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT NULL,
    FOREIGN KEY (product_id) REFERENCES products(product_id),
    FOREIGN KEY (user_id) REFERENCES users(user_id)
);

-- Wishlist table
CREATE TABLE wishlist (
    wishlist_id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    product_id INT NOT NULL,
    added_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(user_id),
    FOREIGN KEY (product_id) REFERENCES products(product_id),
    UNIQUE (user_id, product_id) -- Prevent duplicate wishlist entries
);

-- Cart table
CREATE TABLE cart (
    cart_id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    product_id INT NOT NULL,
    quantity INT NOT NULL CHECK (quantity > 0),
    added_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(user_id),
    FOREIGN KEY (product_id) REFERENCES products(product_id),
    UNIQUE (user_id, product_id) -- Prevent duplicate cart entries
);

-- Orders table
CREATE TABLE orders (
    order_id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    total_amount DECIMAL(10,2) NOT NULL,
    status ENUM('pending', 'processing', 'shipped', 'delivered', 'cancelled') DEFAULT 'pending',
    shipping_address TEXT NOT NULL,
    billing_address TEXT,
    payment_method VARCHAR(50) NOT NULL,
    payment_status ENUM('pending', 'paid', 'failed') DEFAULT 'pending',
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT NULL,
    FOREIGN KEY (user_id) REFERENCES users(user_id)
);

-- Order Items table
CREATE TABLE order_items (
    order_item_id INT PRIMARY KEY AUTO_INCREMENT,
    order_id INT NOT NULL,
    product_id INT NOT NULL,
    quantity INT NOT NULL CHECK (quantity > 0),
    price DECIMAL(10,2) NOT NULL,
    FOREIGN KEY (order_id) REFERENCES orders(order_id),
    FOREIGN KEY (product_id) REFERENCES products(product_id)
);

-- Sales table
CREATE TABLE sales (
    sale_id INT PRIMARY KEY AUTO_INCREMENT,
    order_id INT NOT NULL,
    vendor_id INT,
    product_id INT NOT NULL,
    quantity INT NOT NULL,
    total DECIMAL(10,2) NOT NULL,
    sale_date DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (order_id) REFERENCES orders(order_id),
    FOREIGN KEY (vendor_id) REFERENCES vendors(vendor_id),
    FOREIGN KEY (product_id) REFERENCES products(product_id)
);
CREATE TABLE tokens (
    token_id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    token VARCHAR(64) UNIQUE NOT NULL,
    expires_at DATETIME NOT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(user_id) ON DELETE CASCADE
);
```

---

### Detailed Explanation of Additions and Features
1. **User Registration**: Kept simple with `email`, `password`, `first_name`, `last_name`, and optional `phone`.
2. **Admins and Vendors**: Separate tables for role clarity. Vendors have an `approved` flag and are tied to an admin via `created_by`.
3. **Products**: 
   - Supports multiple images via `product_images` (up to 5 total, with one `primary_image` in `products`).
   - Includes `average_rating` and `review_count` for aggregated review stats.
4. **Product Reviews**: Users can rate (1-5) and leave optional text reviews, tied to both `users` and `products`.
5. **Wishlist**: Users can save products for later, with a unique constraint to avoid duplicates.
6. **Cart**: Temporary storage for items, with a unique constraint to prevent duplicate products per user.
7. **Orders**: Enhanced with `payment_method`, `payment_status`, and separate `shipping_address`/`billing_address` fields.
8. **Order Items**: Captures historical pricing and quantities per order.
9. **Sales**: Tracks revenue per product/vendor for reporting purposes.
10. **Constraints**: Foreign keys and checks (e.g., `quantity > 0`, `rating BETWEEN 1 AND 5`) ensure data integrity.




ALTER TABLE vendors
DROP FOREIGN KEY vendors_ibfk_1;

ALTER TABLE vendors
ADD CONSTRAINT vendors_ibfk_1
FOREIGN KEY (created_by) REFERENCES users(user_id)
ON DELETE SET NULL ON UPDATE CASCADE;
