<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *'); // Adjust for production
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

require_once '../config/database.php';

// Get the database connection
$dbConfig = new DatabaseConfig();
$conn = $dbConfig->getConnection();

try {
    // Get the raw POST data
    $data = json_decode(file_get_contents('php://input'), true);

    // Validate required fields
    if (!isset($data['first_name']) || !isset($data['last_name']) || !isset($data['email']) || !isset($data['phone']) || !isset($data['password'])) {
        echo json_encode(['success' => false, 'message' => 'All fields are required']);
        exit;
    }

    // Step 1: Trim and sanitize inputs
    $first_name = trim($data['first_name']);
    $last_name = trim($data['last_name']);
    $email = trim($data['email']);
    $phone = trim($data['phone']);
    $password = trim($data['password']);

    // Step 2: Sanitize strings to prevent XSS
    $first_name = htmlspecialchars($first_name, ENT_QUOTES, 'UTF-8');
    $last_name = htmlspecialchars($last_name, ENT_QUOTES, 'UTF-8');
    $email = filter_var($email, FILTER_SANITIZE_EMAIL);
    $phone = filter_var($phone, FILTER_SANITIZE_STRING);

    // Step 3: Validate email format
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        echo json_encode(['success' => false, 'message' => 'Invalid email format']);
        exit;
    }

    // Step 4: Validate phone number (must be 10 digits, starting with 0)
    if (!preg_match('/^0[0-9]{9}$/', $phone)) {
        echo json_encode(['success' => false, 'message' => 'Invalid phone number format (e.g., 0791333586)']);
        exit;
    }

    // Step 5: Validate first_name and last_name (e.g., only letters and spaces)
    if (!preg_match('/^[A-Za-z\s]+$/', $first_name) || !preg_match('/^[A-Za-z\s]+$/', $last_name)) {
        echo json_encode(['success' => false, 'message' => 'First name and last name can only contain letters and spaces']);
        exit;
    }

    // Step 6: Check if email already exists
    $stmt = $conn->prepare("SELECT user_id FROM users WHERE email = ?");
    $stmt->execute([$email]);
    if ($stmt->rowCount() > 0) {
        echo json_encode(['success' => false, 'message' => 'Email already exists']);
        exit;
    }

    // Step 7: Hash the password
    $hashed_password = password_hash($password, PASSWORD_DEFAULT);
    if ($hashed_password === false) {
        echo json_encode(['success' => false, 'message' => 'Error hashing password']);
        exit;
    }

    // Step 8: Insert user into the database
    $stmt = $conn->prepare("INSERT INTO users (email, password, first_name, last_name, phone, role, created_at) VALUES (?, ?, ?, ?, ?, ?, NOW())");
    $role = 'user'; // Default role
    $stmt->execute([$email, $hashed_password, $first_name, $last_name, $phone, $role]);

    echo json_encode(['success' => true, 'message' => 'Registration successful']);
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
}
?>