<?php
// Ensure no whitespace before the opening tag
header('Content-Type: application/json');
session_start();

// Suppress warnings and notices (for production)
error_reporting(E_ALL & ~E_WARNING & ~E_NOTICE);
ini_set('display_errors', 0);

// Include database connection
require_once "../config/database.php";

// Check if user is authenticated
if (!isset($_SESSION['user_id'])) {
    echo json_encode(['success' => false, 'message' => 'Please log in to update your password.']);
    exit;
}

$userId = $_SESSION['user_id'];
$db = new DatabaseConfig();
$conn = $db->getConnection();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $currentPassword = trim($_POST['current_password'] ?? '');
    $newPassword = trim($_POST['new_password'] ?? '');
    $confirmPassword = trim($_POST['confirm_password'] ?? '');

    // Basic validation
    if (empty($currentPassword) || empty($newPassword) || empty($confirmPassword)) {
        echo json_encode(['success' => false, 'message' => 'All password fields are required.']);
        exit;
    }

    if ($newPassword !== $confirmPassword) {
        echo json_encode(['success' => false, 'message' => 'New password and confirm password do not match.']);
        exit;
    }

    if (strlen($newPassword) < 8) {
        echo json_encode(['success' => false, 'message' => 'New password must be at least 8 characters long.']);
        exit;
    }

    try {
        // Fetch the current password hash
        $stmt = $conn->prepare("SELECT password FROM users WHERE user_id = :user_id");
        $stmt->execute(['user_id' => $userId]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$user) {
            echo json_encode(['success' => false, 'message' => 'User not found.']);
            exit;
        }

        // Verify the current password
        if (!password_verify($currentPassword, $user['password'])) {
            echo json_encode(['success' => false, 'message' => 'Current password is incorrect.']);
            exit;
        }

        // Hash the new password
        $newPasswordHash = password_hash($newPassword, PASSWORD_DEFAULT);

        // Update the password
        $stmt = $conn->prepare("UPDATE users SET password = :password, updated_at = NOW() WHERE user_id = :user_id");
        $stmt->execute([
            'password' => $newPasswordHash,
            'user_id' => $userId
        ]);

        echo json_encode(['success' => true, 'message' => 'Password updated successfully!']);
    } catch (PDOException $e) {
        error_log("Failed to update password: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'An error occurred while updating your password. Please try again.']);
    }
} else {
    echo json_encode(['success' => false, 'message' => 'Invalid request method.']);
}

exit;