<?php
require_once 'token_check.php'; // Ensures authentication and sets $userId
require_once __DIR__ . '/../../backend/classes/Database.php';
require_once __DIR__ . '/../../backend/classes/Auth.php'; // Include Auth class for isAdmin()
require_once __DIR__ . '/../../backend/classes/Product.php';
require_once __DIR__ . '/../../backend/classes/Order.php';
require_once __DIR__ . '/../../backend/classes/Vendor.php';

$db = (new Database())->getConnection();
$auth = new Auth(); // Instantiate Auth for isAdmin check
$product = new Product();
$order = new Order();
$vendor = new Vendor();

// Check if the request is for a static file and serve it directly
$requestUri = $_SERVER['REQUEST_URI'];
$filename = __DIR__ . parse_url($requestUri, PHP_URL_PATH);
if (preg_match('/\.(jpg|jpeg|png|gif|css|js)$/', $filename) && file_exists($filename)) {
    return false; // Let the server handle static files
}

// Validate admin status using Auth class
$isAdmin = $auth->isAdmin($userId); // Use Auth class for isAdmin check

// Fetch products data
$products = $product->getAllProducts($userId, false, $isAdmin); // Adjust isVendor to false if not applicable
$totalProducts = is_array($products) ? count($products) : 0;
$newProducts = is_array($products) ? array_reduce($products, function ($count, $p) {
    return $count + (isNewProduct($p['created_at']) ? 1 : 0);
}, 0) : 0;
$lowStockItems = is_array($products) ? array_reduce($products, function ($count, $p) {
    return $count + (($p['stock'] ?? 0) < 10 ? 1 : 0);
}, 0) : 0;

// Fetch orders data (use getOrders with admin override)
$orders = $isAdmin ? $order->getOrders(null, true) : $order->getOrders($userId, $isAdmin); // Use null user_id for admins
$pendingOrders = is_array($orders) ? array_reduce($orders, function ($count, $o) {
    return $count + (strtolower($o['status'] ?? '') === 'pending' ? 1 : 0);
}, 0) : 0;
$totalSales = is_array($orders) ? array_reduce($orders, function ($sum, $o) {
    return $sum + floatval($o['total_amount'] ?? 0);
}, 0) : 0;
$salesGrowth = $totalSales > 0 ? number_format($totalSales / 1000 * 30, 0) : 0; // Placeholder 30% growth
$totalIncome = $totalSales; // Using total sales from orders as proxy for income
$incomeIncrease = $totalIncome > 0 ? number_format($totalIncome / 1000 * 22, 0) : 0; // Placeholder 22% increase

function isNewProduct($createdAt) {
    if (!$createdAt) return false;
    $created = new DateTime($createdAt);
    $now = new DateTime();
    $interval = $now->diff($created);
    return $interval->days <= 7; // Within 7 days
}
?>

<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width initial-scale=1.0">
    <title>RoamingNdovu | Dashboard</title>
    <!-- GLOBAL MAINLY STYLES-->
    <link href="./assets/vendors/bootstrap/dist/css/bootstrap.min.css" rel="stylesheet" />
    <link href="./assets/vendors/font-awesome/css/font-awesome.min.css" rel="stylesheet" />
    <link href="./assets/vendors/themify-icons/css/themify-icons.css" rel="stylesheet" />
    <!-- PLUGINS STYLES-->
    <link href="./assets/vendors/jvectormap/jquery-jvectormap-2.0.3.css" rel="stylesheet" />
    <!-- THEME STYLES-->
    <link href="assets/css/main.min.css" rel="stylesheet" />
    <!-- PAGE LEVEL STYLES-->
    <link rel="icon" href="../Media/logo.png" type="image/x-icon">
</head>

<body class="fixed-navbar">
    <div class="page-wrapper">
        <?php include 'header-sidebar.php'; ?>

        <div class="content-wrapper">
            <!-- START PAGE CONTENT-->
            <div class="page-content fade-in-up">
                <div class="row">
                    <div class="col-lg-3 col-md-6">
                        <div class="ibox bg-success color-white widget-stat">
                            <div class="ibox-body">
                                <h2 class="m-b-5 font-strong" id="upcomingEvents">0</h2>
                                <div class="m-b-5">UPCOMING EVENTS</div><i class="ti-calendar widget-stat-icon"></i>
                                <div><i class="fa fa-level-up m-r-5"></i><small id="eventsIncrease">N/A</small></div>
                            </div>
                        </div>
                    </div>
                    <div class="col-lg-3 col-md-6">
                        <div class="ibox bg-info color-white widget-stat">
                            <div class="ibox-body">
                                <h2 class="m-b-5 font-strong" id="totalPosts">0</h2>
                                <div class="m-b-5">POSTS</div><i class="ti-bar-chart widget-stat-icon"></i>
                                <div><i class="fa fa-level-up m-r-5"></i><small id="postsIncrease">N/A</small></div>
                            </div>
                        </div>
                    </div>
                    <div class="col-lg-3 col-md-6">
                        <div class="ibox bg-warning color-white widget-stat">
                            <div class="ibox-body">
                                <h2 class="m-b-5 font-strong" id="totalIncome"><?php echo '$' . number_format($totalIncome, 2); ?></h2>
                                <div class="m-b-5">TOTAL INCOME</div><i class="fa fa-money widget-stat-icon"></i>
                                <div><i class="fa fa-level-up m-r-5"></i><small id="incomeIncrease"><?php echo $incomeIncrease . '% higher'; ?></small></div>
                            </div>
                        </div>
                    </div>
                    <div class="col-lg-3 col-md-6">
                        <div class="ibox bg-danger color-white widget-stat">
                            <div class="ibox-body">
                                <h2 class="m-b-5 font-strong" id="engagements">0</h2>
                                <div class="m-b-5">ENGAGEMENTS</div><i class="ti-user widget-stat-icon"></i>
                                <div><i class="fa fa-level-down m-r-5"></i><small id="engagementDecrease">N/A</small></div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- PRODUCT CARDS -->
                <div class="page-content fade-in-up">
                    <div class="row">
                        <div class="col-lg-3 col-md-6">
                            <div class="ibox bg-warning color-white widget-stat">
                                <div class="ibox-body">
                                    <h2 class="m-b-5 font-strong" id="totalProducts"><?php echo $totalProducts; ?></h2>
                                    <div class="m-b-5">TOTAL PRODUCTS</div><i class="ti-package widget-stat-icon"></i>
                                    <div><i class="fa fa-level-up m-r-5"></i><small id="newProducts"><?php echo '+' . $newProducts . ' New Arrivals'; ?></small></div>
                                </div>
                            </div>
                        </div>
                        <div class="col-lg-3 col-md-6">
                            <div class="ibox bg-danger color-white widget-stat">
                                <div class="ibox-body">
                                    <h2 class="m-b-5 font-strong" id="pendingOrders"><?php echo $pendingOrders; ?></h2>
                                    <div class="m-b-5">PENDING ORDERS</div><i class="ti-receipt widget-stat-icon"></i>
                                    <div><i class="fa fa-level-up m-r-5"></i><small id="ordersIncrease"><?php echo $pendingOrders ? number_format(($pendingOrders / (count($orders) || 1) * 100), 0) . '% increase' : 'N/A'; ?></small></div>
                                </div>
                            </div>
                        </div>
                        <div class="col-lg-3 col-md-6">
                            <div class="ibox bg-success color-white widget-stat">
                                <div class="ibox-body">
                                    <h2 class="m-b-5 font-strong" id="totalSales"><?php echo '$' . number_format($totalSales, 2); ?></h2>
                                    <div class="m-b-5">TOTAL SALES</div><i class="fa fa-shopping-cart widget-stat-icon"></i>
                                    <div><i class="fa fa-level-up m-r-5"></i><small id="salesGrowth"><?php echo $salesGrowth . '% Growth'; ?></small></div>
                                </div>
                            </div>
                        </div>
                        <div class="col-lg-3 col-md-6">
                            <div class="ibox bg-info color-white widget-stat">
                                <div class="ibox-body">
                                    <h2 class="m-b-5 font-strong" id="lowStockItems"><?php echo $lowStockItems; ?></h2>
                                    <div class="m-b-5">LOW STOCK ITEMS</div><i class="ti-alert widget-stat-icon"></i>
                                    <div><i class="fa fa-level-down m-r-5"></i><small id="restockNeeded"><?php echo $lowStockItems ? number_format(($lowStockItems / ($totalProducts || 1) * -8), 0) . '% Restock Needed' : 'N/A'; ?></small></div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <div class="col-lg-12">
                    <div class="ibox recent-posts">
                        <div class="ibox-head">
                            <div class="ibox-title">Recent Posts</div>
                            <div class="ibox-tools">
                                <a class="ibox-collapse"><i class="fa fa-minus"></i></a>
                                <a class="dropdown-toggle" data-toggle="dropdown"><i class="fa fa-ellipsis-v"></i></a>
                                <div class="dropdown-menu dropdown-menu-right">
                                    <a class="dropdown-item">View All</a>
                                    <a class="dropdown-item">New Post</a>
                                </div>
                            </div>
                        </div>
                        <div class="ibox-body">
                            <table class="table table-striped table-hover">
                                <thead>
                                    <tr>
                                        <th>Post ID</th>
                                        <th>Title</th>
                                        <th>Author</th>
                                        <th>Category</th>
                                        <th>Status</th>
                                        <th width="150px">Date</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <!-- Placeholder, no posts table -->
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>

                <div class="col-lg-12">
                    <div class="ibox recent-posts">
                        <div class="ibox-head">
                            <div class="ibox-title">Recent Orders</div>
                            <div class="ibox-tools">
                                <a class="ibox-collapse"><i class="fa fa-minus"></i></a>
                                <a class="dropdown-toggle" data-toggle="dropdown"><i class="fa fa-ellipsis-v"></i></a>
                                <div class="dropdown-menu dropdown-menu-right">
                                    <a class="dropdown-item" href="orders.php">View All Orders</a>
                                    <a class="dropdown-item">New Order</a>
                                </div>
                            </div>
                        </div>
                        <div class="ibox-body">
                            <table class="table table-striped table-hover" id="ordersTable">
                                <thead>
                                    <tr>
                                        <th>Order ID</th>
                                        <th>Customer</th>
                                        <th>Product</th>
                                        <th>Total Amount</th>
                                        <th>Status</th>
                                        <th width="150px">Date</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php
                                    if (is_array($orders)) {
                                        foreach (array_slice($orders, 0, 5) as $order) {
                                            $statusClass = strtolower($order['status'] ?? '') === 'completed' ? 'success' : (strtolower($order['status'] ?? '') === 'pending' ? 'warning' : 'danger');
                                            $productName = 'N/A'; // Placeholder, fetch from order_items if needed
                                            echo "<tr>
                                                <td><a href='order_details.html'>{$order['order_id']}</a></td>
                                                <td>@User{$order['user_id']}</td>
                                                <td>{$productName}</td>
                                                <td>\${" . number_format($order['total_amount'] ?? 0, 2) . "}</td>
                                                <td><span class='badge badge-{$statusClass}'>" . htmlspecialchars($order['status'] ?? 'Unknown') . "</span></td>
                                                <td>" . (new DateTime($order['created_at'] ?? 'now'))->format('m/d/Y') . "</td>
                                            </tr>";
                                        }
                                    }
                                    ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>

                <style>
                    .visitors-table tbody tr td:last-child {
                        display: flex;
                        align-items: center;
                    }

                    .visitors-table .progress {
                        flex: 1;
                    }

                    .visitors-table .progress-parcent {
                        text-align: right;
                        margin-left: 10px;
                    }
                </style>

            <!-- END PAGE CONTENT-->
            <footer class="page-footer">
                <div class="font-13">
                    <span id="currentYear"></span> © <b>SpaceTechDevs</b> - All rights reserved.
                </div>
                <div class="to-top"><i class="fa fa-angle-double-up"></i></div>
            </footer>
        </div>
    </div>
    <!-- BEGIN THEME CONFIG PANEL-->
    <div class="theme-config">
        <div class="theme-config-toggle"><i class="fa fa-cog theme-config-show"></i><i class="ti-close theme-config-close"></i></div>
        <div class="theme-config-box">
            <div class="text-center font-18 m-b-20">SETTINGS</div>
            <div class="font-strong">LAYOUT OPTIONS</div>
            <div class="check-list m-b-20 m-t-10">
                <label class="ui-checkbox ui-checkbox-gray">
                    <input id="_fixedNavbar" type="checkbox" checked>
                    <span class="input-span"></span>Fixed navbar</label>
                <label class="ui-checkbox ui-checkbox-gray">
                    <input id="_fixedlayout" type="checkbox">
                    <span class="input-span"></span>Fixed layout</label>
                <label class="ui-checkbox ui-checkbox-gray">
                    <input class="js-sidebar-toggler" type="checkbox">
                    <span class="input-span"></span>Collapse sidebar</label>
            </div>
            <div class="font-strong">LAYOUT STYLE</div>
            <div class="m-t-10">
                <label class="ui-radio ui-radio-gray m-r-10">
                    <input type="radio" name="layout-style" value="" checked="">
                    <span class="input-span"></span>Fluid</label>
                <label class="ui-radio ui-radio-gray">
                    <input type="radio" name="layout-style" value="1">
                    <span class="input-span"></span>Boxed</label>
            </div>
            <div class="m-t-10 m-b-10 font-strong">THEME COLORS</div>
            <div class="d-flex m-b-20">
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Default">
                    <label>
                        <input type="radio" name="setting-theme" value="default" checked="">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-white"></div>
                        <div class="color-small bg-ebony"></div>
                    </label>
                </div>
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Blue">
                    <label>
                        <input type="radio" name="setting-theme" value="blue">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-blue"></div>
                        <div class="color-small bg-ebony"></div>
                    </label>
                </div>
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Green">
                    <label>
                        <input type="radio" name="setting-theme" value="green">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-green"></div>
                        <div class="color-small bg-ebony"></div>
                    </label>
                </div>
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Purple">
                    <label>
                        <input type="radio" name="setting-theme" value="purple">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-purple"></div>
                        <div class="color-small bg-ebony"></div>
                    </label>
                </div>
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Orange">
                    <label>
                        <input type="radio" name="setting-theme" value="orange">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-orange"></div>
                        <div class="color-small bg-ebony"></div>
                    </label>
                </div>
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Pink">
                    <label>
                        <input type="radio" name="setting-theme" value="pink">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-pink"></div>
                        <div class="color-small bg-ebony"></div>
                    </label>
                </div>
            </div>
            <div class="d-flex">
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="White">
                    <label>
                        <input type="radio" name="setting-theme" value="white">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color"></div>
                        <div class="color-small bg-silver-100"></div>
                    </label>
                </div>
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Blue light">
                    <label>
                        <input type="radio" name="setting-theme" value="blue-light">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-blue"></div>
                        <div class="color-small bg-silver-100"></div>
                    </label>
                </div>
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Green light">
                    <label>
                        <input type="radio" name="setting-theme" value="green-light">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-green"></div>
                        <div class="color-small bg-silver-100"></div>
                    </label>
                </div>
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Purple light">
                    <label>
                        <input type="radio" name="setting-theme" value="purple-light">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-purple"></div>
                        <div class="color-small bg-silver-100"></div>
                    </label>
                </div>
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Orange light">
                    <label>
                        <input type="radio" name="setting-theme" value="orange-light">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-orange"></div>
                        <div class="color-small bg-silver-100"></div>
                    </label>
                </div>
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Pink light">
                    <label>
                        <input type="radio" name="setting-theme" value="pink-light">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-pink"></div>
                        <div class="color-small bg-silver-100"></div>
                    </label>
                </div>
            </div>
        </div>
    </div>
    <!-- END THEME CONFIG PANEL-->
    <!-- BEGIN PAGA BACKDROPS-->
    <div class="sidenav-backdrop backdrop"></div>
    <div class="preloader-backdrop">
        <div class="page-preloader">Loading</div>
    </div>
    <!-- END PAGA BACKDROPS-->
    <!-- CORE PLUGINS-->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js" integrity="sha256-/xUj+3OJU5yExlq6GSYGSHk7tPXikynS7ogEvDej/m4=" crossorigin="anonymous"></script>
    <script src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.11.6/dist/umd/popper.min.js" integrity="sha384-oBqDVmMz9ATKxIep9tiCxS/Z9fNfEXiDAYTujMAeBAsjFuCZSmKbSSUnQlmh/jp3" crossorigin="anonymous"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js" integrity="sha384-geWF76RCwLtnZ8qwWowPQNguL3RmwHVBC9FhGdlKrxdiJJigb/j/68SIy3Te4Bkz" crossorigin="anonymous"></script>
    <script src="./assets/vendors/metisMenu/dist/metisMenu.min.js" type="text/javascript"></script>
    <script src="./assets/vendors/jquery-slimscroll/jquery.slimscroll.min.js" type="text/javascript"></script>
    <!-- PAGE LEVEL PLUGINS-->
    <script src="./assets/vendors/DataTables/datatables.min.js" type="text/javascript"></script>
    <!-- CORE SCRIPTS-->
    <script src="assets/js/app.min.js" type="text/javascript"></script>
    <!-- PAGE LEVEL SCRIPTS-->
    <script type="text/javascript">
        $(document).ready(function() {
            // Initialize DataTable for orders
            $('#ordersTable').DataTable({
                pageLength: 5,
                responsive: true,
                "order": [[5, "desc"]], // Sort by date descending
                "searching": false,
                "paging": false,
                "info": false
            });

            // Live search functionality (if needed for other tables)
            $('#searchInput').on('keyup', function() {
                $('#usersTable').DataTable().search(this.value).draw();
            });
        });
    </script>
</body>
</html>