<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

require_once 'token_check.php'; // Ensures authentication and sets $tokenFromCookie, $userId
require_once __DIR__ . '/../../backend/classes/Database.php';
require_once __DIR__ . '/../../backend/classes/Auth.php';
require_once __DIR__ . '/../../backend/classes/Order.php';
require_once __DIR__ . '/../../backend/classes/Vendor.php';

$db = (new Database())->getConnection();
$auth = new Auth(); // Instantiate Auth for role checks
$order = new Order();
$vendor = new Vendor();

$isVendor = $auth->isVendor($userId); // Use Auth class for isVendor
$isAdmin = $auth->isAdmin($userId);   // Use Auth class for isAdmin

// Fetch all orders with basic details
try {
    $orders = $order->getOrders($userId, $isAdmin);
} catch (PDOException $e) {
    error_log("Database error in orders: " . $e->getMessage());
    $orders = []; // Fallback to empty array if query fails
}

// Handle order creation
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'create_order') {
    $shippingAddress = $_POST['shipping_address'] ?? '';
    $paymentMethod = $_POST['payment_method'] ?? '';

    if (empty($shippingAddress) || empty($paymentMethod)) {
        $error = 'Shipping address and payment method are required';
    } else {
        $result = $order->createOrder($userId, $shippingAddress, $paymentMethod);
        if (isset($result['order_id'])) {
            header("Location: orders.php?success=Order created successfully (Order ID: {$result['order_id']})");
            exit();
        } else {
            $error = $result['error'] ?? 'Failed to create order';
        }
    }
}

// Handle order deletion
if ($_SERVER['REQUEST_METHOD'] === 'GET' && isset($_GET['action']) && $_GET['action'] === 'delete_order' && isset($_GET['order_id'])) {
    $orderId = $_GET['order_id'];
    $result = $order->deleteOrder($orderId, $userId, $isAdmin);
    if (isset($result['message'])) {
        header("Location: orders.php?success={$result['message']}");
        exit();
    } else {
        $error = $result['error'] ?? 'Failed to cancel order';
    }
}
?>

<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>RoamingNdovu | Dashboard</title>
    <!-- GLOBAL MAINLY STYLES-->
    <link href="./assets/vendors/bootstrap/dist/css/bootstrap.min.css" rel="stylesheet" />
    <link href="./assets/vendors/font-awesome/css/font-awesome.min.css" rel="stylesheet" />
    <link href="./assets/vendors/themify-icons/css/themify-icons.css" rel="stylesheet" />
    <!-- THEME STYLES-->
    <link href="assets/css/main.min.css" rel="stylesheet" />
    <link rel="icon" href="../Media/logo.png" type="image/x-icon">
    <style>
        .modal .form-group {
            margin-bottom: 1rem;
        }
        .modal .btn-primary {
            margin-right: 10px;
        }
        .modal-header .close {
            margin-top: -10px;
            font-size: 1.5rem;
            color: #000;
            opacity: 0.5;
            border: none;
            background: none;
            padding: 0;
            cursor: pointer;
        }
        .modal-header .close:hover {
            opacity: 0.8;
        }
        .alert {
            margin-bottom: 1rem;
            padding: 0.75rem 1.25rem;
            border-radius: 0.25rem;
        }
        .alert-success {
            color: #155724;
            background-color: #d4edda;
            border-color: #c3e6cb;
        }
        .alert-danger {
            color: #721c24;
            background-color: #f8d7da;
            border-color: #f5c6cb;
        }
        #viewOrderModal .modal-dialog,
        #deleteOrderModal .modal-dialog,
        #addOrderModal .modal-dialog {
            max-width: 600px;
        }
        @media (max-width: 768px) {
            .table-responsive {
                overflow-x: auto;
            }
            .btn-sm {
                padding: 5px;
                font-size: 12px;
            }
        }
    </style>
</head>

<body class="fixed-navbar">
    <div class="page-wrapper">
        <?php include 'header-sidebar.php'; ?>

        <div class="content-wrapper">
            <!-- START PAGE CONTENT-->
            <div class="page-heading">
                <h1 class="page-title">Orders</h1>
                <ol class="breadcrumb">
                    <li class="breadcrumb-item">
                        <a href="index.html"><i class="la la-home font-20"></i></a>
                    </li>
                    <li class="breadcrumb-item">Orders</li>
                </ol>
            </div>

            <!-- CUSTOMER ORDERS SECTION -->
            <div class="page-content fade-in-up">
                <div class="row">
                    <div class="col-lg-12">
                        <div class="ibox">
                            <div class="ibox-head">
                                <div class="ibox-title">Customer Orders</div>
                            </div>
                            <div class="ibox-body">
                                <!-- Action Bar with Add Order Button -->
                                <div class="d-flex justify-content-end mb-3">
                                    <button class="btn btn-success" data-bs-toggle="modal" data-bs-target="#addOrderModal"><i class="fa fa-plus"></i> Add Order</button>
                                </div>

                                <!-- Display error or success message if any -->
                                <?php if (isset($error)) echo "<div class='alert alert-danger'>{$error}</div>"; ?>
                                <?php if (isset($_GET['success'])) echo "<div class='alert alert-success'>{$_GET['success']}</div>"; ?>

                                <!-- ORDERS TABLE -->
                                <div class="table-responsive">
                                    <table class="table table-hover">
                                        <thead>
                                            <tr>
                                                <th>#</th>
                                                <th>Order ID</th>
                                                <th>Total Amount</th>
                                                <th>Status</th>
                                                <th>Order Date</th>
                                                <th>Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php
                                            $counter = 1;
                                            foreach ($orders as $order) {
                                                $statusBadge = '';
                                                switch (strtolower($order['status'] ?? 'pending')) {
                                                    case 'completed':
                                                        $statusBadge = '<span class="badge badge-success">Completed</span>';
                                                        break;
                                                    case 'pending':
                                                        $statusBadge = '<span class="badge badge-warning">Pending</span>';
                                                        break;
                                                    case 'cancelled':
                                                        $statusBadge = '<span class="badge badge-danger">Cancelled</span>';
                                                        break;
                                                    default:
                                                        $statusBadge = '<span class="badge badge-info">Processing</span>';
                                                }

                                                echo "<tr>
                                                    <td>{$counter}</td>
                                                    <td>{$order['order_id']}</td>
                                                    <td>\${$order['total_amount'] }</td>
                                                    <td>{$statusBadge}</td>
                                                    <td>{$order['created_at']}</td>
                                                    <td>
                                                        <button class='btn btn-sm btn-primary view-order-btn' data-bs-toggle='modal' data-bs-target='#viewOrderModal' data-order-id='{$order['order_id']}'>
                                                            <i class='fa fa-eye'></i>
                                                        </button>
                                                        <button class='btn btn-sm btn-danger delete-order-btn' data-bs-toggle='modal' data-bs-target='#deleteOrderModal' data-order-id='{$order['order_id']}'>
                                                            <i class='fa fa-trash'></i>
                                                        </button>
                                                    </td>
                                                </tr>";
                                                $counter++;
                                            }
                                            ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- View Order Modal -->
            <div class="modal fade" id="viewOrderModal" tabindex="-1" aria-labelledby="viewOrderModalLabel" aria-hidden="true">
                <div class="modal-dialog">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title" id="viewOrderModalLabel">View Order</h5>
                            <button type="button" class="close" data-bs-dismiss="modal" aria-label="Close">
                                <span aria-hidden="true">×</span>
                            </button>
                        </div>
                        <div class="modal-body">
                            <?php
                            if (isset($_GET['view_order_id'])) {
                                $orderId = $_GET['view_order_id'];
                                $order = $order->getOrder($orderId, $userId, $isAdmin);
                                if ($order) {
                                    $userStmt = $db->prepare("SELECT first_name, last_name FROM users WHERE user_id = :user_id");
                                    $userStmt->execute([':user_id' => $order['user_id']]);
                                    $user = $userStmt->fetch(PDO::FETCH_ASSOC);
                                    $customerName = $user ? $user['first_name'] . ' ' . $user['last_name'] : 'Unknown';

                                    $itemsStmt = $db->prepare("SELECT oi.quantity, p.name AS product_name, oi.price 
                                                              FROM order_items oi 
                                                              JOIN products p ON oi.product_id = p.product_id 
                                                              WHERE oi.order_id = :order_id");
                                    $itemsStmt->execute([':order_id' => $orderId]);
                                    $items = $itemsStmt->fetchAll(PDO::FETCH_ASSOC);
                                    $productsList = '';
                                    foreach ($items as $item) {
                                        $itemTotal = $item['price'] * $item['quantity'];
                                        $productsList .= "<li>{$item['quantity']} x {$item['product_name']} - \${$itemTotal}</li>";
                                    }

                                    echo "<div class='form-group'>
                                            <label>Order ID</label>
                                            <p>{$order['order_id']}</p>
                                          </div>
                                          <div class='form-group'>
                                            <label>Customer Name</label>
                                            <p>{$customerName}</p>
                                          </div>
                                          <div class='form-group'>
                                            <label>Products</label>
                                            <ul>{$productsList}</ul>
                                          </div>
                                          <div class='form-group'>
                                            <label>Total Price</label>
                                            <p>\${$order['total_amount']}</p>
                                          </div>
                                          <div class='form-group'>
                                            <label>Shipping Address</label>
                                            <p>{$order['shipping_address']}</p>
                                          </div>
                                          <div class='form-group'>
                                            <label>Payment Method</label>
                                            <p>{$order['payment_method']}</p>
                                          </div>
                                          <div class='form-group'>
                                            <label>Status</label>
                                            <p>" . getStatusBadge($order['status'] ?? 'pending') . "</p>
                                          </div>
                                          <div class='form-group'>
                                            <label>Order Date</label>
                                            <p>{$order['created_at']}</p>
                                          </div>";
                                } else {
                                    echo "<div class='alert alert-danger'>Order not found</div>";
                                }
                            }
                            ?>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Delete Order Modal -->
            <div class="modal fade" id="deleteOrderModal" tabindex="-1" aria-labelledby="deleteOrderModalLabel" aria-hidden="true">
                <div class="modal-dialog">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title" id="deleteOrderModalLabel">Confirm Delete</h5>
                            <button type="button" class="close" data-bs-dismiss="modal" aria-label="Close">
                                <span aria-hidden="true">×</span>
                            </button>
                        </div>
                        <div class="modal-body">
                            <?php
                            if (isset($_GET['delete_order_id'])) {
                                $orderId = $_GET['delete_order_id'];
                                $order = $order->getOrder($orderId, $userId, $isAdmin);
                                if ($order) {
                                    echo "<p>Are you sure you want to cancel order #{$order['order_id']}?</p>";
                                } else {
                                    echo "<div class='alert alert-danger'>Order not found</div>";
                                }
                            }
                            ?>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                            <a href="orders.php?action=delete_order&order_id=<?php echo isset($_GET['delete_order_id']) ? $_GET['delete_order_id'] : ''; ?>" class="btn btn-danger">Confirm Cancel</a>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Add Order Modal -->
            <div class="modal fade" id="addOrderModal" tabindex="-1" aria-labelledby="addOrderModalLabel" aria-hidden="true">
                <div class="modal-dialog">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title" id="addOrderModalLabel">Create New Order</h5>
                            <button type="button" class="close" data-bs-dismiss="modal" aria-label="Close">
                                <span aria-hidden="true">×</span>
                            </button>
                        </div>
                        <div class="modal-body">
                            <form method="POST" action="orders.php">
                                <input type="hidden" name="action" value="create_order">
                                <div class="form-group">
                                    <label for="shippingAddress">Shipping Address</label>
                                    <input type="text" class="form-control" id="shippingAddress" name="shipping_address" required>
                                </div>
                                <div class="form-group">
                                    <label for="paymentMethod">Payment Method</label>
                                    <select class="form-control" id="paymentMethod" name="payment_method" required>
                                        <option value="Credit Card">Credit Card</option>
                                        <option value="PayPal">PayPal</option>
                                        <option value="Bank Transfer">Bank Transfer</option>
                                    </select>
                                </div>
                                <button type="submit" class="btn btn-primary">Create Order</button>
                            </form>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- SCRIPTS -->
            <script src="./assets/vendors/jquery/dist/jquery.min.js"></script>
            <script src="./assets/vendors/popper.js/dist/umd/popper.min.js"></script>
            <script src="./assets/vendors/bootstrap/dist/js/bootstrap.min.js"></script>
            <script src="./assets/vendors/metisMenu/dist/metisMenu.min.js"></script>
            <script src="./assets/vendors/jquery-slimscroll/jquery.slimscroll.min.js"></script>
            <script src="assets/js/app.min.js"></script>
            <script>
                $(document).ready(function() {
                    // View order modal
                    $('.view-order-btn').on('click', function() {
                        const orderId = $(this).data('order-id');
                        window.location.href = 'orders.php?view_order_id=' + orderId;
                    });

                    // Delete order modal
                    $('.delete-order-btn').on('click', function() {
                        const orderId = $(this).data('order-id');
                        window.location.href = 'orders.php?delete_order_id=' + orderId;
                    });
                });

                function getStatusBadge(status) {
                    switch (status.toLowerCase()) {
                        case 'completed':
                            return '<span class="badge badge-success">Completed</span>';
                        case 'pending':
                            return '<span class="badge badge-warning">Pending</span>';
                        case 'cancelled':
                            return '<span class="badge badge-danger">Cancelled</span>';
                        default:
                            return '<span class="badge badge-info">Processing</span>';
                    }
                }
            </script>
        </body>
</html>