<?php
require_once 'token_check.php'; // Ensures authentication and sets $tokenFromCookie, $userId
require_once __DIR__ . '/../../backend/classes/Database.php';
require_once __DIR__ . '/../../backend/classes/Auth.php';
require_once __DIR__ . '/../../backend/classes/Product.php';
require_once __DIR__ . '/../../backend/classes/Vendor.php';

$db = (new Database())->getConnection();
$auth = new Auth(); // Instantiate Auth for role checks
$product = new Product();
$vendor = new Vendor();

$isVendor = $auth->isVendor($userId); // Use Auth class for isVendor
$isAdmin = $auth->isAdmin($userId);   // Use Auth class for isAdmin

// Fetch all products with debug and additional images
try {
    $products = $product->getAllProducts($userId, $isVendor, $isAdmin);
    foreach ($products as &$productItem) {
        $productId = $productItem['product_id'];
        $stmt = $db->prepare("SELECT image_url FROM product_images WHERE product_id = :product_id");
        $stmt->execute([':product_id' => $productId]);
        $productItem['additional_images'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
        error_log("Fetched additional images for product_id $productId: " . json_encode($productItem['additional_images']));
    }
    unset($productItem); // Unset reference to avoid issues
} catch (PDOException $e) {
    error_log("Database error in products: " . $e->getMessage());
    $products = []; // Fallback to empty array if query fails
}

// Fetch all vendors for the dropdown
try {
    $vendorsStmt = $db->query("SELECT vendor_id, vendor_name FROM vendors WHERE is_active = 1");
    $vendors = $vendorsStmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Database error in vendors: " . $e->getMessage());
    $vendors = []; // Fallback to empty array if query fails
}

// Fetch all categories for the dropdown
$categoryStmt = $db->query("SELECT category_id, name FROM categories");
$categories = $categoryStmt->fetchAll(PDO::FETCH_ASSOC);

// Handle form submission for editing products (direct database update)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'edit_product') {
    $productId = $_POST['product_id'];
    $data = [
        'name' => $_POST['name'],
        'description' => $_POST['description'],
        'price' => $_POST['price'],
        'stock' => $_POST['stock'],
        'primary_image' => $_POST['primary_image_hidden'], // Use hidden field for existing image
        'category_id' => $_POST['category_id']
    ];

    // Handle primary image upload if new file is provided
    if ($_FILES['primary_image']['name']) {
        $targetDir = __DIR__ . '/uploads/';
        $primaryImage = $_FILES['primary_image']['name'];
        $targetFile = $targetDir . basename($primaryImage);
        $imageFileType = strtolower(pathinfo($targetFile, PATHINFO_EXTENSION));
        $allowedTypes = ['jpg', 'jpeg', 'png', 'gif'];
        if (in_array($imageFileType, $allowedTypes) && move_uploaded_file($_FILES['primary_image']['tmp_name'], $targetFile)) {
            $data['primary_image'] = '/dashboard/admin/uploads/' . $primaryImage;
        } else {
            $error = 'Invalid or failed primary image upload';
        }
    }

    // Handle additional images
    $additionalImages = [];
    if (isset($_FILES['images'])) {
        for ($i = 0; $i < count($_FILES['images']['name']); $i++) {
            $imageName = $_FILES['images']['name'][$i];
            $targetFile = $targetDir . basename($imageName);
            $imageFileType = strtolower(pathinfo($targetFile, PATHINFO_EXTENSION));
            if (in_array($imageFileType, $allowedTypes) && move_uploaded_file($_FILES['images']['tmp_name'][$i], $targetFile)) {
                $additionalImages[] = '/dashboard/admin/uploads/' . $imageName;
            }
        }
    }

    if (!isset($error)) {
        $result = $product->updateProduct($productId, $data, $userId, $isVendor, $isAdmin);
        if ($result && isset($result['message'])) {
            // Update additional images
            if (!empty($additionalImages)) {
                $stmt = $db->prepare("DELETE FROM product_images WHERE product_id = :product_id");
                $stmt->execute([':product_id' => $productId]);
                foreach ($additionalImages as $imageUrl) {
                    $stmt = $db->prepare("INSERT INTO product_images (product_id, image_url) VALUES (:product_id, :image_url)");
                    $stmt->execute([':product_id' => $productId, ':image_url' => $imageUrl]);
                }
            }
            header("Location: products.php?success=" . urlencode($result['message']));
            exit();
        } else {
            $error = $result['error'] ?? 'Failed to update product';
        }
    }
}

// Handle add product form submission with direct database insertion
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add_product') {
    $targetDir = __DIR__ . '/uploads/'; // Directory to store uploaded images
    if (!file_exists($targetDir)) {
        mkdir($targetDir, 0777, true);
    }

    $name = $_POST['name'];
    $description = $_POST['description'];
    $price = $_POST['price'];
    $stock = $_POST['stock'];
    $categoryId = $_POST['category_id'];
    $vendorId = $_POST['vendor_id'] ?: null;

    // Handle primary image upload
    $primaryImage = $_FILES['primary_image']['name'];
    $targetFile = $targetDir . basename($primaryImage);
    $imageFileType = strtolower(pathinfo($targetFile, PATHINFO_EXTENSION));
    $allowedTypes = ['jpg', 'jpeg', 'png', 'gif'];
    if (in_array($imageFileType, $allowedTypes) && move_uploaded_file($_FILES['primary_image']['tmp_name'], $targetFile)) {
        $primaryImage = '/dashboard/admin/uploads/' . $primaryImage;
    } else {
        $error = 'Invalid or failed primary image upload';
    }

    // Handle additional images
    $additionalImages = [];
    if (isset($_FILES['images'])) {
        for ($i = 0; $i < count($_FILES['images']['name']); $i++) {
            $imageName = $_FILES['images']['name'][$i];
            $targetFile = $targetDir . basename($imageName);
            $imageFileType = strtolower(pathinfo($targetFile, PATHINFO_EXTENSION));
            if (in_array($imageFileType, $allowedTypes) && move_uploaded_file($_FILES['images']['tmp_name'][$i], $targetFile)) {
                $additionalImages[] = '/dashboard/admin/uploads/' . $imageName;
            }
        }
    }

    if (!isset($error)) {
        $data = [
            'name' => $name,
            'description' => $description,
            'price' => $price,
            'stock' => $stock,
            'primary_image' => $primaryImage,
            'category_id' => $categoryId,
            'vendor_id' => $vendorId
        ];

        // Direct call to Product class
        $result = $product->createProduct($data, $userId, $isVendor, $isAdmin);
        if ($result && isset($result['product_id'])) {
            $productId = $result['product_id'];
            foreach ($additionalImages as $imageUrl) {
                $stmt = $db->prepare("INSERT INTO product_images (product_id, image_url) VALUES (:product_id, :image_url)");
                $stmt->execute([':product_id' => $productId, ':image_url' => $imageUrl]);
            }
            header("Location: products.php?success=" . urlencode($result['message'] ?? 'Product added successfully'));
            exit();
        } else {
            $error = $result['error'] ?? 'Failed to add product';
        }
    }
}

// Handle add category form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add_category') {
    $name = $_POST['name'];
    $description = $_POST['description'];
    $stmt = $db->prepare("INSERT INTO categories (name, description, created_at) VALUES (:name, :description, NOW())");
    if ($stmt->execute([':name' => $name, ':description' => $description])) {
        header("Location: products.php?success=Category added successfully");
        exit();
    } else {
        $error = 'Failed to add category';
    }
}
?>

<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>RoamingNdovu | Dashboard</title>
    <!-- GLOBAL MAINLY STYLES-->
    <link href="./assets/vendors/bootstrap/dist/css/bootstrap.min.css" rel="stylesheet" />
    <link href="./assets/vendors/font-awesome/css/font-awesome.min.css" rel="stylesheet" />
    <link href="./assets/vendors/themify-icons/css/themify-icons.css" rel="stylesheet" />
    <!-- PLUGINS STYLES-->
    <link href="./assets/vendors/DataTables/datatables.min.css" rel="stylesheet" />
    <!-- THEME STYLES-->
    <link href="assets/css/main.min.css" rel="stylesheet" />
    <link rel="icon" href="../Media/logo.png" type="image/x-icon">
    <style>
        .modal .form-group {
            margin-bottom: 1rem;
        }
        .modal .btn-primary {
            margin-right: 10px;
        }
        .modal-header .close {
            margin-top: -10px;
            font-size: 1.5rem;
            color: #000;
            opacity: 0.5;
            border: none;
            background: none;
            padding: 0;
            cursor: pointer;
        }
        .modal-header .close:hover {
            opacity: 0.8;
        }
        .disabled-action {
            pointer-events: none;
            opacity: 0.6;
        }
        .alert {
            margin-bottom: 1rem;
            padding: 0.75rem 1.25rem;
            border-radius: 0.25rem;
        }
        .alert-success {
            color: #155724;
            background-color: #d4edda;
            border-color: #c3e6cb;
        }
        .alert-danger {
            color: #721c24;
            background-color: #f8d7da;
            border-color: #f5c6cb;
        }
        #editModal .modal-dialog,
        #viewModal .modal-dialog,
        #addProductModal .modal-dialog,
        #addCategoryModal .modal-dialog {
            max-width: 800px; /* Wider for image gallery */
        }
        #deleteModal .modal-dialog {
            max-width: 400px;
        }
        .product-img, .modal-image {
            width: 150px;
            height: 150px;
            object-fit: cover;
            border: 2px solid #ddd;
            border-radius: 5px;
            margin: 5px;
            cursor: pointer;
            transition: transform 0.2s;
        }
        .product-img:hover, .modal-image:hover {
            transform: scale(1.05);
        }
        .image-gallery, .modal-image-gallery {
            display: flex;
            flex-wrap: wrap;
            justify-content: center;
        }
        .image-preview {
            display: flex;
            flex-wrap: wrap;
            margin-top: 10px;
        }
        .image-preview img {
            max-width: 100px;
            height: auto;
            margin-right: 10px;
            border: 2px solid #ddd;
            border-radius: 5px;
        }
        @media (max-width: 768px) {
            .table-responsive {
                overflow-x: auto;
            }
            .btn-sm {
                padding: 5px;
                font-size: 12px;
            }
            .product-img, .modal-image {
                width: 100px;
                height: 100px;
            }
            .image-gallery img, .modal-image-gallery img {
                max-width: 80px;
            }
            .image-preview img {
                max-width: 80px;
            }
        }
    </style>
</head>

<body class="fixed-navbar">
    <div class="page-wrapper">
        <?php include 'header-sidebar.php'; ?>

        <div class="content-wrapper">
            <!-- START PAGE CONTENT-->
            <div class="page-heading">
                <h1 class="page-title">Products</h1>
                <ol class="breadcrumb">
                    <li class="breadcrumb-item">
                        <a href="index.html"><i class="la la-home font-20"></i></a>
                    </li>
                    <li class="breadcrumb-item">Products</li>
                </ol>
            </div>
            <div class="page-content fade-in-up">
                <!-- Search Bar and Add Buttons -->
                <div class="mb-3">
                    <input class="form-control w-25 mb-2" id="searchInput" type="text" placeholder="Search products...">
                    <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addProductModal"><i class="fa fa-plus"></i> Add New Product</button>
                    <button class="btn btn-success ms-2" data-bs-toggle="modal" data-bs-target="#addCategoryModal"><i class="fa fa-plus"></i> Add Category</button>
                </div>

                <!-- Display error or success message if any -->
                <?php if (isset($error)) echo "<div class='alert alert-danger'>{$error}</div>"; ?>
                <?php if (isset($_GET['success'])) echo "<div class='alert alert-success'>{$_GET['success']}</div>"; ?>

                <!-- Responsive Table Wrapper -->
                <div class="table-responsive">
                    <table class="table table-striped table-hover" id="productsTable">
                        <thead class="table-dark">
                            <tr>
                                <th>#</th>
                                <th>Product Image</th>
                                <th>Product Name</th>
                                <th>Category</th>
                                <th>Price</th>
                                <th>Stock</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            $counter = 1;
                            foreach ($products as $product) {
                                $isCurrentVendor = ($product['vendor_id'] == $userId && $isVendor && !$isAdmin);
                                $editDisabled = $isCurrentVendor ? '' : ' disabled-action';
                                $deleteDisabled = $isCurrentVendor ? '' : ' disabled-action';
                                echo "<tr>
                                    <td>{$counter}</td>
                                    <td>
                                        <img src='{$product['primary_image']}' class='product-img' onclick=\"expandImage(this, '{$product['product_id']}')\">
                                    </td>
                                    <td>{$product['name']}</td>
                                    <td>{$product['category_name']}</td>
                                    <td>\${$product['price']}</td>
                                    <td>{$product['stock']}</td>
                                    <td>
                                        <button class='btn btn-info btn-sm view-btn' data-bs-toggle='modal' data-bs-target='#viewModal' data-product-id='{$product['product_id']}'>View</button>
                                        <button class='btn btn-primary btn-sm edit-btn' data-bs-toggle='modal' data-bs-target='#editModal' data-product-id='{$product['product_id']}'>Edit</button>
                                        <button class='btn btn-danger btn-sm delete-btn' data-product-id='{$product['product_id']}'>Delete</button>
                                    </td>
                                </tr>";
                                $counter++;
                            }
                            ?>
                        </tbody>
                        <style>
                            .product-img{

                                width: 50px;
                                height: 50px;
                                object-fit: cover;
                                border: 2px solid #ddd;
                                border-radius: 5px;
                                margin: -5px;
                                cursor: pointer;
                                transition: transform 0.2s;
                                border-radius: 50%;
                            }
                            

                        </style>
                    </table>
                </div>

                <!-- Add Product Modal -->
                <div class="modal fade" id="addProductModal" tabindex="-1" aria-labelledby="addProductModalLabel" aria-hidden="true">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="addProductModalLabel">Add New Product</h5>
                                <button type="button" class="close" data-bs-dismiss="modal" aria-label="Close">
                                    <span aria-hidden="true">×</span>
                                </button>
                            </div>
                            <div class="modal-body">
                                <form method="POST" action="products.php" id="addProductForm" enctype="multipart/form-data">
                                    <input type="hidden" name="action" value="add_product">
                                    <div class="form-group">
                                        <label for="addName">Product Name <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" id="addName" name="name" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="addDescription">Description <span class="text-danger">*</span></label>
                                        <textarea class="form-control" id="addDescription" name="description" required></textarea>
                                    </div>
                                    <div class="form-group">
                                        <label for="addPrice">Price <span class="text-danger">*</span></label>
                                        <input type="number" step="0.01" class="form-control" id="addPrice" name="price" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="addStock">Stock <span class="text-danger">*</span></label>
                                        <input type="number" class="form-control" id="addStock" name="stock" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="addPrimaryImage">Primary Image <span class="text-danger">*</span></label>
                                        <input type="file" class="form-control" id="addPrimaryImage" name="primary_image" accept="image/*" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="addCategoryId">Category <span class="text-danger">*</span></label>
                                        <select class="form-control" id="addCategoryId" name="category_id" required>
                                            <?php
                                            foreach ($categories as $category) {
                                                echo "<option value='{$category['category_id']}'>{$category['name']}</option>";
                                            }
                                            ?>
                                        </select>
                                    </div>
                                    <div class="form-group">
                                        <label for="addVendorId">Vendor (optional for admins)</label>
                                        <select class="form-control" id="addVendorId" name="vendor_id">
                                            <option value="">None</option>
                                            <?php foreach ($vendors as $v) {
                                                echo "<option value='{$v['vendor_id']}'>{$v['vendor_name']}</option>";
                                            } ?>
                                        </select>
                                    </div>
                                    <div class="form-group" id="addImageGallery">
                                        <label>Additional Images (up to 4)</label>
                                        <input type="file" class="form-control add-image-input" name="images[]" accept="image/*">
                                        <button type="button" class="btn btn-secondary mt-2" id="addImageField">Add Another Image</button>
                                        <div class="image-preview mt-2" id="addImagePreview"></div>
                                    </div>
                                    <button type="submit" class="btn btn-primary">Save Product</button>
                                </form>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Add Category Modal -->
                <div class="modal fade" id="addCategoryModal" tabindex="-1" aria-labelledby="addCategoryModalLabel" aria-hidden="true">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="addCategoryModalLabel">Add New Category</h5>
                                <button type="button" class="close" data-bs-dismiss="modal" aria-label="Close">
                                    <span aria-hidden="true">×</span>
                                </button>
                            </div>
                            <div class="modal-body">
                                <form method="POST" action="products.php" id="addCategoryForm">
                                    <input type="hidden" name="action" value="add_category">
                                    <div class="form-group">
                                        <label for="addCategoryName">Category Name <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" id="addCategoryName" name="name" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="addCategoryDescription">Description</label>
                                        <textarea class="form-control" id="addCategoryDescription" name="description"></textarea>
                                    </div>
                                    <button type="submit" class="btn btn-primary">Save Category</button>
                                </form>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- View Modal -->
                <div class="modal fade" id="viewModal" tabindex="-1" aria-labelledby="viewModalLabel" aria-hidden="true">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="viewModalLabel">View Product</h5>
                                <button type="button" class="close" data-bs-dismiss="modal" aria-label="Close">
                                    <span aria-hidden="true">×</span>
                                </button>
                            </div>
                            <div class="modal-body">
                                <div class="form-group">
                                    <label>Primary Image</label>
                                    <img id="viewPrimaryImage" class="modal-image" src="">
                                </div>
                                <div class="form-group">
                                    <label>Additional Images</label>
                                    <div id="viewImageGallery" class="modal-image-gallery">
                                        <?php
                                        $productId = isset($_GET['product_id']) ? $_GET['product_id'] : (isset($_POST['product_id']) ? $_POST['product_id'] : null);
                                        if ($productId) {
                                            $stmt = $db->prepare("SELECT image_url FROM product_images WHERE product_id = :product_id");
                                            $stmt->execute([':product_id' => $productId]);
                                            $additionalImages = $stmt->fetchAll(PDO::FETCH_ASSOC);
                                            foreach ($additionalImages as $image) {
                                                echo "<img src='{$image['image_url']}' alt='Additional Image' class='modal-image' onclick=\"expandImage(this)\">";
                                            }
                                        }
                                        ?>
                                    </div>
                                </div>
                                <div class="form-group">
                                    <label>Product Name</label>
                                    <p id="viewName"></p>
                                </div>
                                <div class="form-group">
                                    <label>Description</label>
                                    <p id="viewDescription"></p>
                                </div>
                                <div class="form-group">
                                    <label>Price</label>
                                    <p id="viewPrice"></p>
                                </div>
                                <div class="form-group">
                                    <label>Stock</label>
                                    <p id="viewStock"></p>
                                </div>
                                <div class="form-group">
                                    <label>Category</label>
                                    <p id="viewCategory"></p>
                                </div>
                                <div class="form-group">
                                    <label>Vendor</label>
                                    <p id="viewVendor"></p>
                                </div>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Edit Modal -->
                <div class="modal fade" id="editModal" tabindex="-1" aria-labelledby="editModalLabel" aria-hidden="true">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="editModalLabel">Edit Product</h5>
                                <button type="button" class="close" data-bs-dismiss="modal" aria-label="Close">
                                    <span aria-hidden="true">×</span>
                                </button>
                            </div>
                            <div class="modal-body">
                                <form method="POST" action="products.php" id="editProductForm" enctype="multipart/form-data">
                                    <input type="hidden" name="action" value="edit_product">
                                    <input type="hidden" id="editProductId" name="product_id">
                                    <div class="form-group">
                                        <label for="editName">Product Name <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" id="editName" name="name" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="editDescription">Description <span class="text-danger">*</span></label>
                                        <textarea class="form-control" id="editDescription" name="description" required></textarea>
                                    </div>
                                    <div class="form-group">
                                        <label for="editPrice">Price <span class="text-danger">*</span></label>
                                        <input type="number" step="0.01" class="form-control" id="editPrice" name="price" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="editStock">Stock <span class="text-danger">*</span></label>
                                        <input type="number" class="form-control" id="editStock" name="stock" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="editPrimaryImage">Primary Image <span class="text-danger">*</span></label>
                                        <input type="file" class="form-control" id="editPrimaryImage" name="primary_image" accept="image/*">
                                        <input type="hidden" id="editPrimaryImageHidden" name="primary_image_hidden">
                                        <img id="editPrimaryImagePreview" class="modal-image" src="" style="display: none;">
                                    </div>
                                    <div class="form-group">
                                        <label for="editCategoryId">Category <span class="text-danger">*</span></label>
                                        <select class="form-control" id="editCategoryId" name="category_id" required>
                                            <?php
                                            foreach ($categories as $category) {
                                                echo "<option value='{$category['category_id']}'>{$category['name']}</option>";
                                            }
                                            ?>
                                        </select>
                                    </div>
                                    <div class="form-group">
                                        <label for="editVendorId">Vendor (optional for admins)</label>
                                        <select class="form-control" id="editVendorId" name="vendor_id">
                                            <option value="">None</option>
                                            <?php foreach ($vendors as $v) {
                                                echo "<option value='{$v['vendor_id']}'>{$v['vendor_name']}</option>";
                                            } ?>
                                        </select>
                                    </div>
                                    <div class="form-group" id="editImageGallery">
                                        <label>Additional Images (up to 4)</label>
                                        <input type="file" class="form-control edit-image-input" name="images[]" accept="image/*">
                                        <button type="button" class="btn btn-secondary mt-2" id="editAddImageField">Add Another Image</button>
                                        <div class="modal-image-gallery mt-2" id="editImagePreview"></div>
                                    </div>
                                    <button type="submit" class="btn btn-primary">Save Changes</button>
                                </form>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Delete Modal -->
                <div class="modal fade" id="deleteModal" tabindex="-1" aria-labelledby="deleteModalLabel" aria-hidden="true">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="deleteModalLabel">Confirm Delete</h5>
                                <button type="button" class="close" data-bs-dismiss="modal" aria-label="Close">
                                    <span aria-hidden="true">×</span>
                                </button>
                            </div>
                            <div class="modal-body">
                                <p>Are you sure you want to delete this product? This action cannot be undone.</p>
                                <input type="hidden" id="deleteProductId">
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                <button type="button" class="btn btn-danger" id="confirmDelete">Delete</button>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Image Modal -->
                <div id="imageModal" class="modal fade" tabindex="-1">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title">Product Image</h5>
                                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                            </div>
                            <div class="modal-body text-center">
                                <img id="modalImage" class="modal-image">
                            </div>
                        </div>
                    </div>
                </div>

                <!-- END PAGE CONTENT-->
                <footer class="page-footer">
                    <div class="font-13">
                        <span id="currentYear"></span> © <b>SpaceTechDevs</b> - All rights reserved.
                    </div>
                    <div class="to-top"><i class="fa fa-angle-double-up"></i></div>
                </footer>
            </div>
        </div>
    </div>
    <!-- BEGIN THEME CONFIG PANEL-->
    <div class="theme-config">
        <div class="theme-config-toggle"><i class="fa fa-cog theme-config-show"></i><i class="ti-close theme-config-close"></i></div>
        <div class="theme-config-box">
            <div class="text-center font-18 m-b-20">SETTINGS</div>
            <div class="font-strong">LAYOUT OPTIONS</div>
            <div class="check-list m-b-20 m-t-10">
                <label class="ui-checkbox ui-checkbox-gray">
                    <input id="_fixedNavbar" type="checkbox" checked>
                    <span class="input-span"></span>Fixed navbar</label>
                <label class="ui-checkbox ui-checkbox-gray">
                    <input id="_fixedlayout" type="checkbox">
                    <span class="input-span"></span>Fixed layout</label>
                <label class="ui-checkbox ui-checkbox-gray">
                    <input class="js-sidebar-toggler" type="checkbox">
                    <span class="input-span"></span>Collapse sidebar</label>
            </div>
            <div class="font-strong">LAYOUT STYLE</div>
            <div class="m-t-10">
                <label class="ui-radio ui-radio-gray m-r-10">
                    <input type="radio" name="layout-style" value="" checked="">
                    <span class="input-span"></span>Fluid</label>
                <label class="ui-radio ui-radio-gray">
                    <input type="radio" name="layout-style" value="1">
                    <span class="input-span"></span>Boxed</label>
            </div>
            <div class="m-t-10 m-b-10 font-strong">THEME COLORS</div>
            <div class="d-flex m-b-20">
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Default">
                    <label>
                        <input type="radio" name="setting-theme" value="default" checked="">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-white"></div>
                        <div class="color-small bg-ebony"></div>
                    </label>
                </div>
                <!-- Add other color options as needed -->
            </div>
        </div>
    </div>
    <!-- CORE PLUGINS-->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js" integrity="sha256-/xUj+3OJU5yExlq6GSYGSHk7tPXikynS7ogEvDej/m4=" crossorigin="anonymous"></script>
    <script src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.11.6/dist/umd/popper.min.js" integrity="sha384-oBqDVmMz9ATKxIep9tiCxS/Z9fNfEXiDAYTujMAeBAsjFuCZSmKbSSUnQlmh/jp3" crossorigin="anonymous"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js" integrity="sha384-geWF76RCwLtnZ8qwWowPQNguL3RmwHVBC9FhGdlKrxdiJJigb/j/68SIy3Te4Bkz" crossorigin="anonymous"></script>
    <script src="./assets/vendors/metisMenu/dist/metisMenu.min.js" type="text/javascript"></script>
    <script src="./assets/vendors/jquery-slimscroll/jquery.slimscroll.min.js" type="text/javascript"></script>
    <!-- PAGE LEVEL PLUGINS-->
    <script src="./assets/vendors/DataTables/datatables.min.js" type="text/javascript"></script>
    <!-- CORE SCRIPTS-->
    <script src="assets/js/app.min.js" type="text/javascript"></script>
    <!-- PAGE LEVEL SCRIPTS-->
    <script type="text/javascript">
        $(document).ready(function() {
            // Initialize DataTable
            $('#productsTable').DataTable({
                pageLength: 10,
                responsive: true,
                "order": [[0, "asc"]], // Sort by # column ascending by default
                "searching": true,
                "dom": '<"top"lf>rt<"bottom"ip>',
                "language": {
                    "search": "Search:"
                }
            });

            // Live search functionality
            $('#searchInput').on('keyup', function() {
                $('#productsTable').DataTable().search(this.value).draw();
            });

            // Initialize modals
            var addProductModal = new bootstrap.Modal(document.getElementById('addProductModal'), { keyboard: false });
            var viewModal = new bootstrap.Modal(document.getElementById('viewModal'), { keyboard: false });
            var editModal = new bootstrap.Modal(document.getElementById('editModal'), { keyboard: false });
            var deleteModal = new bootstrap.Modal(document.getElementById('deleteModal'), { keyboard: false });
            var addCategoryModal = new bootstrap.Modal(document.getElementById('addCategoryModal'), { keyboard: false });
            var imageModal = new bootstrap.Modal(document.getElementById('imageModal'), { keyboard: false });

            // Function to expand image
            window.expandImage = function(img) {
                document.getElementById('modalImage').src = img.src;
                imageModal.show();
            };

            // Preview images on file selection
            $('#addPrimaryImage, .add-image-input, #editPrimaryImage, .edit-image-input').on('change', function(e) {
                const files = e.target.files;
                const preview = $(this).nextAll('.image-preview, .modal-image-gallery').first();
                preview.empty();
                for (let i = 0; i < files.length; i++) {
                    const reader = new FileReader();
                    reader.onload = function(e) {
                        preview.append('<img src="' + e.target.result + '" alt="Preview" class="modal-image">');
                    };
                    reader.readAsDataURL(files[i]);
                }
                if ($(this).attr('id') === 'editPrimaryImage') {
                    $('#editPrimaryImagePreview').attr('src', e.target.result).show();
                }
            });

            // View functionality
            $('.view-btn').on('click', function() {
                const productId = $(this).data('product-id');
                const product = <?php echo json_encode($products); ?>.find(p => p.product_id == productId);
                if (!product) {
                    alert('Product not found');
                    return;
                }

                // Populate view modal with PHP-fetched data
                $('#viewPrimaryImage').attr('src', product.primary_image || '');
                $('#viewName').text(product.name || 'N/A');
                $('#viewDescription').text(product.description || 'N/A');
                $('#viewPrice').text('$' + (product.price || 'N/A'));
                $('#viewStock').text(product.stock || 'N/A');
                $('#viewCategory').text(product.category_name || 'N/A');
                $('#viewVendor').text(product.vendor_name || 'N/A');

                // Update additional images dynamically based on PHP data
                $('#viewImageGallery').empty();
                if (product.additional_images && product.additional_images.length > 0) {
                    product.additional_images.forEach(img => {
                        $('#viewImageGallery').append(
                            `<img src="${img.image_url}" alt="Additional Image" class="modal-image" onclick="expandImage(this)">`
                        );
                    });
                }

                viewModal.show();
            });

            // Edit functionality
            $(document).on('click', '.edit-btn', function() {
                console.log('Edit clicked');
                const productId = $(this).data('product-id');
                const product = <?php echo json_encode($products); ?>.find(p => p.product_id == productId);
                if (!product) {
                    alert('Product not found');
                    return;
                }
                $('#editProductId').val(product.product_id);
                $('#editName').val(product.name || '');
                $('#editDescription').val(product.description || '');
                $('#editPrice').val(product.price || '');
                $('#editStock').val(product.stock || '');
                $('#editPrimaryImageHidden').val(product.primary_image || '');
                $('#editPrimaryImagePreview').attr('src', product.primary_image).show();
                $('#editPrimaryImage').val(''); // Clear file input
                $('#editCategoryId').val(product.category_id || '');
                $('#editVendorId').val(product.vendor_id || '');
                $('#editImagePreview').empty();

                // Fetch additional images for edit (still using AJAX for now, can be replaced with PHP if needed)
                $.ajax({
                    url: '../../backend/api/v1/product_images.php?product_id=' + productId,
                    method: 'GET',
                    headers: { 'Authorization': '<?php echo $tokenFromCookie; ?>' },
                    success: function(images) {
                        $('#editImageGallery').empty();
                        if (Array.isArray(images) && images.length > 0) {
                            images.forEach((img, index) => {
                                if (img.image_url !== product.primary_image || index === 0) {
                                    $('#editImageGallery').prepend('<input type="file" class="form-control edit-image-input mt-2" name="images[]" accept="image/*"><div class="image-preview mt-2" id="editImagePreview' + Date.now() + '"></div>');
                                }
                            });
                            if ($('#editImageGallery .edit-image-input').length < 4) {
                                $('#editImageGallery').append('<button type="button" class="btn btn-secondary mt-2" id="editAddImageField">Add Another Image</button>');
                            }
                        }
                    },
                    error: function(xhr) {
                        console.error('Failed to fetch images:', xhr.status, xhr.responseText);
                    }
                });

                editModal.show();
            });

            // Save Edit (direct database update)
            $('#editProductForm').on('submit', function(e) {
                e.preventDefault();
                let formData = $(this).serializeArray();
                let data = {};
                $.each(formData, function(index, field) {
                    data[field.name] = field.value;
                });

                // Handle primary image upload if new file is provided
                if ($('#editPrimaryImage')[0].files[0]) {
                    let targetDir = '<?php echo __DIR__ . '/uploads/'; ?>';
                    let primaryImage = $('#editPrimaryImage')[0].files[0].name;
                    let targetFile = targetDir + basename(primaryImage);
                    let imageFileType = primaryImage.split('.').pop().toLowerCase();
                    let allowedTypes = ['jpg', 'jpeg', 'png', 'gif'];
                    if ($.inArray(imageFileType, allowedTypes) !== -1) {
                        data['primary_image'] = '/dashboard/admin/uploads/' + primaryImage;
                    } else {
                        alert('Invalid primary image type');
                        return;
                    }
                } else {
                    data['primary_image'] = $('#editPrimaryImageHidden').val();
                }

                // Handle additional images
                let additionalImages = [];
                $('.edit-image-input').each(function() {
                    if (this.files && this.files[0]) {
                        let imageName = this.files[0].name;
                        let imageFileType = imageName.split('.').pop().toLowerCase();
                        if ($.inArray(imageFileType, allowedTypes) !== -1) {
                            additionalImages.push('/dashboard/admin/uploads/' + imageName);
                        }
                    }
                });

                data['product_id'] = $('#editProductId').val();
                $.ajax({
                    url: 'products.php',
                    method: 'POST',
                    data: data,
                    success: function(response) {
                        alert('Product updated successfully');
                        location.reload();
                    },
                    error: function(xhr) {
                        alert('Failed to update product: ' + xhr.responseText);
                    }
                });

                if (additionalImages.length > 0) {
                    let formData = new FormData();
                    formData.append('action', 'edit_product');
                    formData.append('product_id', $('#editProductId').val());
                    $('.edit-image-input').each(function() {
                        if (this.files && this.files[0]) {
                            formData.append('images[]', this.files[0]);
                        }
                    });
                    $.ajax({
                        url: 'products.php',
                        method: 'POST',
                        data: formData,
                        contentType: false,
                        processData: false,
                        success: function(response) {
                            console.log('Additional images updated');
                        },
                        error: function(xhr) {
                            console.error('Failed to update additional images:', xhr.responseText);
                        }
                    });
                }
            });

            // Delete functionality
            $(document).on('click', '.delete-btn', function() {
                console.log('Delete clicked');
                const productId = $(this).data('product-id');
                $('#deleteProductId').val(productId);
                deleteModal.show();
            });

            $('#confirmDelete').on('click', function() {
                const productId = $('#deleteProductId').val();
                $.ajax({
                    url: '../../backend/api/v1/products.php?id=' + productId,
                    method: 'DELETE',
                    headers: { 'Authorization': '<?php echo $tokenFromCookie; ?>' },
                    success: function(response) {
                        const result = JSON.parse(response);
                        alert(result.message || 'Product deleted successfully');
                        location.reload();
                    },
                    error: function(xhr) {
                        if (xhr.status === 405) {
                            $.ajax({
                                url: '../../backend/api/v1/products.php?id=' + productId,
                                method: 'POST',
                                headers: { 'Authorization': '<?php echo $tokenFromCookie; ?>', 'Content-Type': 'application/json' },
                                data: JSON.stringify({ action: 'delete' }),
                                success: function(response) {
                                    const result = JSON.parse(response);
                                    alert(result.message || 'Product deleted successfully');
                                    location.reload();
                                },
                                error: function(xhr) {
                                    alert('Failed to delete product: ' + xhr.responseText);
                                }
                            });
                        } else {
                            alert('Failed to delete product: ' + xhr.responseText);
                        }
                    }
                });
                deleteModal.hide();
            });

            // Add category functionality
            $('#addCategoryForm').on('submit', function(e) {
                e.preventDefault();
                const formData = $(this).serialize();
                $.ajax({
                    url: 'products.php',
                    method: 'POST',
                    data: formData,
                    success: function(response) {
                        alert('Category added successfully');
                        location.reload();
                    },
                    error: function(xhr) {
                        alert('Failed to add category: ' + xhr.responseText);
                    }
                });
            });

            // Dynamic image fields for add modal
            $('#addImageField').on('click', function() {
                if ($('#addImageGallery .add-image-input').length < 4) {
                    $('#addImageGallery').append('<input type="file" class="form-control add-image-input mt-2" name="images[]" accept="image/*"><div class="image-preview mt-2" id="addImagePreview' + Date.now() + '"></div>');
                } else {
                    alert('Maximum 4 additional images allowed.');
                }
            });

            // Dynamic image fields for edit modal
            $('#editAddImageField').on('click', function() {
                if ($('#editImageGallery .edit-image-input').length < 4) {
                    $('#editImageGallery').append('<input type="file" class="form-control edit-image-input mt-2" name="images[]" accept="image/*"><div class="image-preview mt-2" id="editImagePreview' + Date.now() + '"></div>');
                } else {
                    alert('Maximum 4 additional images allowed.');
                }
            });
        });
    </script>
</body>
</html>