<?php
require_once 'token_check.php'; // Ensures authentication and sets $tokenFromCookie
require_once __DIR__ . '/../../backend/classes/Database.php';
require_once __DIR__ . '/../../backend/classes/User.php';

$db = (new Database())->getConnection();
$user = new User();

// Fetch all users with vendor details if applicable
$stmt = $db->query("SELECT u.user_id, u.email, u.first_name, u.last_name, u.phone, u.created_at, u.role, 
                    v.vendor_name, v.bio 
                    FROM users u 
                    LEFT JOIN vendors v ON u.email = v.email");
$users = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Handle form submission for adding users
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'add_user') {
    $email = filter_var($_POST['email'], FILTER_SANITIZE_EMAIL);
    $password = $_POST['password'];
    $firstName = filter_var($_POST['first_name'], FILTER_SANITIZE_STRING);
    $lastName = filter_var($_POST['last_name'], FILTER_SANITIZE_STRING);
    $phone = filter_var($_POST['phone'], FILTER_SANITIZE_STRING);
    $role = $_POST['role'];
    $vendorName = isset($_POST['vendor_name']) ? filter_var($_POST['vendor_name'], FILTER_SANITIZE_STRING) : null;
    $bio = isset($_POST['bio']) ? filter_var($_POST['bio'], FILTER_SANITIZE_STRING) : null;

    if ($email && $password && $firstName && $lastName && $phone && in_array($role, ['admin', 'vendor', 'user'])) {
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
        $db->beginTransaction();
        try {
            // Check for duplicate email
            $checkStmt = $db->prepare("SELECT COUNT(*) FROM users WHERE email = :email");
            $checkStmt->execute([':email' => $email]);
            if ($checkStmt->fetchColumn() > 0) {
                throw new Exception("Email already exists. Please use a different email.");
            }

            // Insert into users table
            $userStmt = $db->prepare("INSERT INTO users (email, password, first_name, last_name, phone, role, created_at) VALUES (:email, :password, :first_name, :last_name, :phone, :role, NOW())");
            $userStmt->execute([
                ':email' => $email,
                ':password' => $hashedPassword,
                ':first_name' => $firstName,
                ':last_name' => $lastName,
                ':phone' => $phone,
                ':role' => $role
            ]);
            $userId = $db->lastInsertId();

            // If vendor, insert into vendors table with created_by as the logged-in admin's user_id
            if ($role === 'vendor') {
                $vendorStmt = $db->prepare("INSERT INTO vendors (email, password, vendor_name, phone, created_at, approved, bio, created_by) VALUES (:email, :password, :vendor_name, :phone, NOW(), 0, :bio, :created_by)");
                $vendorStmt->execute([
                    ':email' => $email,
                    ':password' => $hashedPassword,
                    ':vendor_name' => $vendorName ?? '',
                    ':phone' => $phone,
                    ':bio' => $bio ?? '',
                    ':created_by' => $userId
                ]);
                if (!$vendorName) {
                    header("Location: users.php?success=vendor_added");
                    exit();
                }
            }

            $db->commit();
            $success = "User added successfully!";
            header("Location: users.php?success=" . urlencode($success));
            exit();
        } catch (Exception $e) {
            $db->rollBack();
            $error = match (true) {
                strpos($e->getMessage(), '1062') !== false => "Email already exists. Please use a different email.",
                default => "Failed to add user. Please try again or contact support."
            };
        }
    } else {
        $error = "Invalid input data. Please fill all required fields correctly.";
    }
}

// Handle form submission for editing users
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'edit_user') {
    $userId = $_POST['user_id'];
    $email = filter_var($_POST['email'], FILTER_SANITIZE_EMAIL);
    $password = $_POST['password'] ? password_hash($_POST['password'], PASSWORD_DEFAULT) : null;
    $firstName = filter_var($_POST['first_name'], FILTER_SANITIZE_STRING);
    $lastName = filter_var($_POST['last_name'], FILTER_SANITIZE_STRING);
    $phone = filter_var($_POST['phone'], FILTER_SANITIZE_STRING);
    $role = $_POST['role'];
    $vendorName = isset($_POST['vendor_name']) ? filter_var($_POST['vendor_name'], FILTER_SANITIZE_STRING) : null;
    $bio = isset($_POST['bio']) ? filter_var($_POST['bio'], FILTER_SANITIZE_STRING) : null;

    if ($email && $firstName && $lastName && $phone && in_array($role, ['admin', 'vendor', 'user'])) {
        $db->beginTransaction();
        try {
            // Check for duplicate email (exclude current user)
            $checkStmt = $db->prepare("SELECT COUNT(*) FROM users WHERE email = :email AND user_id != :user_id");
            $checkStmt->execute([':email' => $email, ':user_id' => $userId]);
            if ($checkStmt->fetchColumn() > 0) {
                throw new Exception("Email already exists. Please use a different email.");
            }

            // Update users table
            $fields = [];
            $params = [':user_id' => $userId];
            if ($email) { $fields[] = "email = :email"; $params[':email'] = $email; }
            if ($password) { $fields[] = "password = :password"; $params[':password'] = $password; }
            if ($firstName) { $fields[] = "first_name = :first_name"; $params[':first_name'] = $firstName; }
            if ($lastName) { $fields[] = "last_name = :last_name"; $params[':last_name'] = $lastName; }
            if ($phone) { $fields[] = "phone = :phone"; $params[':phone'] = $phone; }
            if ($role) { $fields[] = "role = :role"; $params[':role'] = $role; }

            if (!empty($fields)) {
                $updateStmt = $db->prepare("UPDATE users SET " . implode(', ', $fields) . " WHERE user_id = :user_id");
                $updateStmt->execute($params);
            }

            // Update vendors table if role is vendor
            if ($role === 'vendor') {
                $vendorStmt = $db->prepare("UPDATE vendors SET vendor_name = :vendor_name, bio = :bio WHERE email = :email");
                $vendorStmt->execute([
                    ':vendor_name' => $vendorName ?? '',
                    ':bio' => $bio ?? '',
                    ':email' => $email
                ]);
            }

            $db->commit();
            $success = "User updated successfully!";
            header("Location: users.php?success=" . urlencode($success));
            exit();
        } catch (Exception $e) {
            $db->rollBack();
            $error = match (true) {
                strpos($e->getMessage(), '1062') !== false => "Email already exists. Please use a different email.",
                default => "Failed to update user. Please try again or contact support."
            };
        }
    } else {
        $error = "Invalid input data. Please fill all required fields correctly.";
    }
}
?>

<!DOCTYPE html>
<html lang="en">

<head>
    <meta charset="UTF-8">
    <meta http-equiv="X-UA-Compatible" content="IE=edge">
    <meta name="viewport" content="width=device-width initial-scale=1.0">
    <title>RoamingNdovu | Dashboard</title>
    <!-- GLOBAL MAINLY STYLES-->
    <link href="./assets/vendors/bootstrap/dist/css/bootstrap.min.css" rel="stylesheet" />
    <link href="./assets/vendors/font-awesome/css/font-awesome.min.css" rel="stylesheet" />
    <link href="./assets/vendors/themify-icons/css/themify-icons.css" rel="stylesheet" />
    <!-- PLUGINS STYLES-->
    <link href="./assets/vendors/DataTables/datatables.min.css" rel="stylesheet" />
    <!-- THEME STYLES-->
    <link href="assets/css/main.min.css" rel="stylesheet" />
    <!-- PAGE LEVEL STYLES-->
    <link rel="icon" href="../Media/logo.png" type="image/x-icon">
    <style>
        .modal .form-group {
            margin-bottom: 1rem;
        }
        .modal .btn-primary {
            margin-right: 10px;
        }
        .modal-header .close {
            margin-top: -10px;
            font-size: 1.5rem;
            color: #000;
            opacity: 0.5;
            border: none;
            background: none;
            padding: 0;
            cursor: pointer;
        }
        .modal-header .close:hover {
            opacity: 0.8;
        }
        .disabled-action {
            pointer-events: none;
            opacity: 0.6;
        }
        .alert {
            margin-bottom: 1rem;
            padding: 0.75rem 1.25rem;
            border-radius: 0.25rem;
        }
        .alert-success {
            color: #155724;
            background-color: #d4edda;
            border-color: #c3e6cb;
        }
        .alert-danger {
            color: #721c24;
            background-color: #f8d7da;
            border-color: #f5c6cb;
        }
        #editModal .modal-dialog,
        #viewModal .modal-dialog,
        #addAdminModal .modal-dialog,
        #addVendorModal .modal-dialog,
        #addCustomerModal .modal-dialog {
            max-width: 500px;
        }
        #deleteModal .modal-dialog {
            max-width: 400px;
        }
    </style>
</head>

<body class="fixed-navbar">
    <div class="page-wrapper">
        <?php include 'header-sidebar.php'; ?>

        <div class="content-wrapper">
            <!-- START PAGE CONTENT-->
            <div class="page-heading">
                <h1 class="page-title">Users</h1>
                <ol class="breadcrumb">
                    <li class="breadcrumb-item">
                        <a href="index.html"><i class="la la-home font-20"></i></a>
                    </li>
                    <li class="breadcrumb-item">Current Users</li>
                </ol>
            </div>
            <div class="page-content fade-in-up">
                <!-- Search Bar -->
                <input class="form-control mb-3" id="searchInput" type="text" placeholder="Search users...">

                <!-- Add User Buttons -->
                <div class="mb-3">
                    <button class="btn btn-success" data-bs-toggle="modal" data-bs-target="#addAdminModal">Add Admin</button>
                    <button class="btn btn-warning" data-bs-toggle="modal" data-bs-target="#addVendorModal">Add Vendor</button>
                    <button class="btn btn-info" data-bs-toggle="modal" data-bs-target="#addCustomerModal">Add Customer</button>
                </div>

                <!-- Display error or success message if any -->
                <?php if (isset($error)) echo "<div class='alert alert-danger'>{$error}</div>"; ?>
                <?php if (isset($_GET['success'])) echo "<div class='alert alert-success'>{$_GET['success']}</div>"; ?>

                <!-- Responsive Table Wrapper -->
                <div class="table-responsive">
                    <table class="table table-striped table-hover" id="usersTable">
                        <thead class="table-dark">
                            <tr>
                                <th>#</th>
                                <th>Name</th>
                                <th>Email</th>
                                <th>Role</th>
                                <th>Phone</th>
                                <th>Created At</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php
                            $counter = 1;
                            $currentUserId = $userId; // From token_check.php
                            foreach ($users as $user) {
                                $fullName = htmlspecialchars($user['first_name'] . ' ' . $user['last_name']);
                                $role = htmlspecialchars($user['role']);
                                $status = ($user['role'] === 'admin' || $user['role'] === 'vendor') ? '<span class="badge bg-success">Active</span>' : '<span class="badge bg-warning text-dark">Pending</span>';
                                $isCurrentAdmin = ($user['user_id'] == $currentUserId && $user['role'] === 'admin');
                                $editDisabled = $isCurrentAdmin ? ' disabled-action' : '';
                                $deleteDisabled = $isCurrentAdmin ? ' disabled-action' : '';
                                echo "<tr>
                                    <td>{$counter}</td>
                                    <td>{$fullName}</td>
                                    <td>{$user['email']}</td>
                                    <td>{$role}</td>
                                    <td>{$user['phone']}</td>
                                    <td>" . (new DateTime($user['created_at']))->format('m/d/Y') . "</td>
                                    <td>
                                        <button class='btn btn-info btn-sm view-btn' data-bs-toggle='modal' data-bs-target='#viewModal' data-user-id='{$user['user_id']}'>View</button>
                                        <button class='btn btn-primary btn-sm edit-btn' data-bs-toggle='modal' data-bs-target='#editModal' data-user-id='{$user['user_id']}'" . ($editDisabled ? ' disabled' : '') . ">Edit</button>
                                        <button class='btn btn-danger btn-sm delete-btn' data-id='{$user['user_id']}'" . ($deleteDisabled ? ' disabled' : '') . ">Delete</button>
                                    </td>
                                </tr>";
                                $counter++;
                            }
                            ?>
                        </tbody>
                    </table>
                </div>

                <!-- Add Admin Modal -->
                <div class="modal fade" id="addAdminModal" tabindex="-1" aria-labelledby="addAdminModalLabel" aria-hidden="true">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="addAdminModalLabel">Add New Admin</h5>
                                <button type="button" class="close" data-bs-dismiss="modal" aria-label="Close">
                                    <span aria-hidden="true">×</span>
                                </button>
                            </div>
                            <div class="modal-body">
                                <form method="POST" action="users.php" id="addAdminForm">
                                    <input type="hidden" name="action" value="add_user">
                                    <div class="form-group">
                                        <label for="adminEmail">Email <span class="text-danger">*</span></label>
                                        <input type="email" class="form-control" name="email" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="adminPassword">Password <span class="text-danger">*</span></label>
                                        <input type="password" class="form-control" name="password" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="adminFirstName">First Name <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" name="first_name" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="adminLastName">Last Name <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" name="last_name" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="adminPhone">Phone <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" name="phone" required>
                                    </div>
                                    <input type="hidden" name="role" value="admin">
                                </form>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                                <button type="submit" form="addAdminForm" class="btn btn-primary">Save Admin</button>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Add Vendor Modal -->
                <div class="modal fade" id="addVendorModal" tabindex="-1" aria-labelledby="addVendorModalLabel" aria-hidden="true">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="addVendorModalLabel">Add New Vendor</h5>
                                <button type="button" class="close" data-bs-dismiss="modal" aria-label="Close">
                                    <span aria-hidden="true">×</span>
                                </button>
                            </div>
                            <div class="modal-body">
                                <form method="POST" action="users.php" id="addVendorForm">
                                    <input type="hidden" name="action" value="add_user">
                                    <div class="form-group">
                                        <label for="vendorEmail">Email <span class="text-danger">*</span></label>
                                        <input type="email" class="form-control" name="email" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="vendorPassword">Password <span class="text-danger">*</span></label>
                                        <input type="password" class="form-control" name="password" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="vendorFirstName">First Name <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" name="first_name" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="vendorLastName">Last Name <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" name="last_name" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="vendorPhone">Phone <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" name="phone" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="vendorName">Vendor Name <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" name="vendor_name" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="vendorBio">Bio</label>
                                        <textarea class="form-control" name="bio" rows="3"></textarea>
                                    </div>
                                    <input type="hidden" name="role" value="vendor">
                                </form>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                                <button type="submit" form="addVendorForm" class="btn btn-primary">Save Vendor</button>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Add Customer Modal -->
                <div class="modal fade" id="addCustomerModal" tabindex="-1" aria-labelledby="addCustomerModalLabel" aria-hidden="true">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="addCustomerModalLabel">Add New Customer</h5>
                                <button type="button" class="close" data-bs-dismiss="modal" aria-label="Close">
                                    <span aria-hidden="true">×</span>
                                </button>
                            </div>
                            <div class="modal-body">
                                <form method="POST" action="users.php" id="addCustomerForm">
                                    <input type="hidden" name="action" value="add_user">
                                    <div class="form-group">
                                        <label for="customerEmail">Email <span class="text-danger">*</span></label>
                                        <input type="email" class="form-control" name="email" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="customerPassword">Password <span class="text-danger">*</span></label>
                                        <input type="password" class="form-control" name="password" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="customerFirstName">First Name <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" name="first_name" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="customerLastName">Last Name <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" name="last_name" required>
                                    </div>
                                    <div class="form-group">
                                        <label for="customerPhone">Phone <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" name="phone" required>
                                    </div>
                                    <input type="hidden" name="role" value="user">
                                </form>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                                <button type="submit" form="addCustomerForm" class="btn btn-primary">Save Customer</button>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- View Modal -->
                <div class="modal fade" id="viewModal" tabindex="-1" aria-labelledby="viewModalLabel" aria-hidden="true">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="viewModalLabel">View User</h5>
                                <button type="button" class="close" data-bs-dismiss="modal" aria-label="Close">
                                    <span aria-hidden="true">×</span>
                                </button>
                            </div>
                            <div class="modal-body">
                                <div class="form-group">
                                    <label>Email</label>
                                    <p id="viewEmail">Loading...</p>
                                </div>
                                <div class="form-group">
                                    <label>First Name</label>
                                    <p id="viewFirstName">Loading...</p>
                                </div>
                                <div class="form-group">
                                    <label>Last Name</label>
                                    <p id="viewLastName">Loading...</p>
                                </div>
                                <div class="form-group">
                                    <label>Phone</label>
                                    <p id="viewPhone">Loading...</p>
                                </div>
                                <div class="form-group">
                                    <label>Role</label>
                                    <p id="viewRole">Loading...</p>
                                </div>
                                <div class="form-group vendor-fields-view" style="display: none;">
                                    <label>Vendor Name</label>
                                    <p id="viewVendorName">Loading...</p>
                                </div>
                                <div class="form-group vendor-fields-view" style="display: none;">
                                    <label>Bio</label>
                                    <p id="viewBio">Loading...</p>
                                </div>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal" id="closeViewModal">Close</button>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Edit Modal -->
                <div class="modal fade" id="editModal" tabindex="-1" aria-labelledby="editModalLabel" aria-hidden="true">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="editModalLabel">Edit User</h5>
                                <button type="button" class="close" data-bs-dismiss="modal" aria-label="Close">
                                    <span aria-hidden="true">×</span>
                                </button>
                            </div>
                            <div class="modal-body">
                                <form method="POST" action="users.php" id="editUserForm">
                                    <input type="hidden" name="action" value="edit_user">
                                    <input type="hidden" id="editUserId" name="user_id">
                                    <div class="form-group">
                                        <label for="editEmail">Email <span class="text-danger">*</span></label>
                                        <input type="email" class="form-control" id="editEmail" name="email" required value="">
                                    </div>
                                    <div class="form-group">
                                        <label for="editPassword">Password (leave blank to keep current)</label>
                                        <input type="password" class="form-control" id="editPassword" name="password" value="">
                                    </div>
                                    <div class="form-group">
                                        <label for="editFirstName">First Name <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" id="editFirstName" name="first_name" required value="">
                                    </div>
                                    <div class="form-group">
                                        <label for="editLastName">Last Name <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" id="editLastName" name="last_name" required value="">
                                    </div>
                                    <div class="form-group">
                                        <label for="editPhone">Phone <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" id="editPhone" name="phone" required value="">
                                    </div>
                                    <div class="form-group">
                                        <label for="editRole">Role <span class="text-danger">*</span></label>
                                        <select class="form-control" id="editRole" name="role" required>
                                            <option value="admin">Admin</option>
                                            <option value="vendor">Vendor</option>
                                            <option value="user">Customer</option>
                                        </select>
                                    </div>
                                    <div class="form-group vendor-fields" style="display: none;">
                                        <label for="editVendorName">Vendor Name <span class="text-danger">*</span></label>
                                        <input type="text" class="form-control" id="editVendorName" name="vendor_name" required value="">
                                    </div>
                                    <div class="form-group vendor-fields" style="display: none;">
                                        <label for="editBio">Bio</label>
                                        <textarea class="form-control" id="editBio" name="bio" rows="3"></textarea>
                                    </div>
                                    <button type="submit" class="btn btn-primary">Save Changes</button>
                                </form>
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Delete Modal -->
                <div class="modal fade" id="deleteModal" tabindex="-1" aria-labelledby="deleteModalLabel" aria-hidden="true">
                    <div class="modal-dialog">
                        <div class="modal-content">
                            <div class="modal-header">
                                <h5 class="modal-title" id="deleteModalLabel">Confirm Delete</h5>
                                <button type="button" class="close" data-bs-dismiss="modal" aria-label="Close">
                                    <span aria-hidden="true">×</span>
                                </button>
                            </div>
                            <div class="modal-body">
                                <p>Are you sure you want to delete this user? This action cannot be undone.</p>
                                <input type="hidden" id="deleteUserId">
                            </div>
                            <div class="modal-footer">
                                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                                <button type="button" class="btn btn-danger" id="confirmDelete">Delete</button>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- END PAGE CONTENT-->
                <footer class="page-footer">
                    <div class="font-13">
                        <span id="currentYear"></span> © <b>SpaceTechDevs</b> - All rights reserved.
                    </div>
                    <div class="to-top"><i class="fa fa-angle-double-up"></i></div>
                </footer>
            </div>
        </div>
    </div>
    <!-- BEGIN THEME CONFIG PANEL-->
    <div class="theme-config">
        <div class="theme-config-toggle"><i class="fa fa-cog theme-config-show"></i><i class="ti-close theme-config-close"></i></div>
        <div class="theme-config-box">
            <div class="text-center font-18 m-b-20">SETTINGS</div>
            <div class="font-strong">LAYOUT OPTIONS</div>
            <div class="check-list m-b-20 m-t-10">
                <label class="ui-checkbox ui-checkbox-gray">
                    <input id="_fixedNavbar" type="checkbox" checked>
                    <span class="input-span"></span>Fixed navbar</label>
                <label class="ui-checkbox ui-checkbox-gray">
                    <input id="_fixedlayout" type="checkbox">
                    <span class="input-span"></span>Fixed layout</label>
                <label class="ui-checkbox ui-checkbox-gray">
                    <input class="js-sidebar-toggler" type="checkbox">
                    <span class="input-span"></span>Collapse sidebar</label>
            </div>
            <div class="font-strong">LAYOUT STYLE</div>
            <div class="m-t-10">
                <label class="ui-radio ui-radio-gray m-r-10">
                    <input type="radio" name="layout-style" value="" checked="">
                    <span class="input-span"></span>Fluid</label>
                <label class="ui-radio ui-radio-gray">
                    <input type="radio" name="layout-style" value="1">
                    <span class="input-span"></span>Boxed</label>
            </div>
            <div class="m-t-10 m-b-10 font-strong">THEME COLORS</div>
            <div class="d-flex m-b-20">
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Default">
                    <label>
                        <input type="radio" name="setting-theme" value="default" checked="">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-white"></div>
                        <div class="color-small bg-ebony"></div>
                    </label>
                </div>
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Blue">
                    <label>
                        <input type="radio" name="setting-theme" value="blue">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-blue"></div>
                        <div class="color-small bg-ebony"></div>
                    </label>
                </div>
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Green">
                    <label>
                        <input type="radio" name="setting-theme" value="green">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-green"></div>
                        <div class="color-small bg-ebony"></div>
                    </label>
                </div>
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Purple">
                    <label>
                        <input type="radio" name="setting-theme" value="purple">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-purple"></div>
                        <div class="color-small bg-ebony"></div>
                    </label>
                </div>
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Orange">
                    <label>
                        <input type="radio" name="setting-theme" value="orange">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-orange"></div>
                        <div class="color-small bg-ebony"></div>
                    </label>
                </div>
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Pink">
                    <label>
                        <input type="radio" name="setting-theme" value="pink">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-pink"></div>
                        <div class="color-small bg-ebony"></div>
                    </label>
                </div>
            </div>
            <div class="d-flex">
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="White">
                    <label>
                        <input type="radio" name="setting-theme" value="white">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color"></div>
                        <div class="color-small bg-silver-100"></div>
                    </label>
                </div>
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Blue light">
                    <label>
                        <input type="radio" name="setting-theme" value="blue-light">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-blue"></div>
                        <div class="color-small bg-silver-100"></div>
                    </label>
                </div>
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Green light">
                    <label>
                        <input type="radio" name="setting-theme" value="green-light">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-green"></div>
                        <div class="color-small bg-silver-100"></div>
                    </label>
                </div>
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Purple light">
                    <label>
                        <input type="radio" name="setting-theme" value="purple-light">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-purple"></div>
                        <div class="color-small bg-silver-100"></div>
                    </label>
                </div>
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Orange light">
                    <label>
                        <input type="radio" name="setting-theme" value="orange-light">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-orange"></div>
                        <div class="color-small bg-silver-100"></div>
                    </label>
                </div>
                <div class="color-skin-box" data-toggle="tooltip" data-original-title="Pink light">
                    <label>
                        <input type="radio" name="setting-theme" value="pink-light">
                        <span class="color-check-icon"><i class="fa fa-check"></i></span>
                        <div class="color bg-pink"></div>
                        <div class="color-small bg-silver-100"></div>
                    </label>
                </div>
            </div>
        </div>
    </div>
    <!-- END THEME CONFIG PANEL-->
    <!-- BEGIN PAGA BACKDROPS-->
    <div class="sidenav-backdrop backdrop"></div>
    <div class="preloader-backdrop">
        <div class="page-preloader">Loading</div>
    </div>
    <!-- END PAGA BACKDROPS-->
    <!-- CORE PLUGINS-->
    <script src="https://code.jquery.com/jquery-3.6.0.min.js" integrity="sha256-/xUj+3OJU5yExlq6GSYGSHk7tPXikynS7ogEvDej/m4=" crossorigin="anonymous"></script>
    <script src="https://cdn.jsdelivr.net/npm/@popperjs/core@2.11.6/dist/umd/popper.min.js" integrity="sha384-oBqDVmMz9ATKxIep9tiCxS/Z9fNfEXiDAYTujMAeBAsjFuCZSmKbSSUnQlmh/jp3" crossorigin="anonymous"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js" integrity="sha384-geWF76RCwLtnZ8qwWowPQNguL3RmwHVBC9FhGdlKrxdiJJigb/j/68SIy3Te4Bkz" crossorigin="anonymous"></script>
    <script src="./assets/vendors/metisMenu/dist/metisMenu.min.js" type="text/javascript"></script>
    <script src="./assets/vendors/jquery-slimscroll/jquery.slimscroll.min.js" type="text/javascript"></script>
    <!-- PAGE LEVEL PLUGINS-->
    <script src="./assets/vendors/DataTables/datatables.min.js" type="text/javascript"></script>
    <!-- CORE SCRIPTS-->
    <script src="assets/js/app.min.js" type="text/javascript"></script>
    <!-- PAGE LEVEL SCRIPTS-->
    <script type="text/javascript">
        $(document).ready(function() {
            // Initialize DataTable
            $('#usersTable').DataTable({
                pageLength: 10,
                responsive: true,
                "order": [[0, "asc"]], // Sort by # column ascending by default
                "searching": true, // Enable search
                "dom": '<"top"lf>rt<"bottom"ip>', // Custom layout with length, filter, table, info, pagination
                "language": {
                    "search": "Search:"
                }
            });

            // Live search functionality
            $('#searchInput').on('keyup', function() {
                $('#usersTable').DataTable().search(this.value).draw();
            });

            // Get token from PHP variable $tokenFromCookie (not used for view/edit now)
            const token = "<?php echo htmlspecialchars($tokenFromCookie ?? '', ENT_QUOTES, 'UTF-8'); ?>";
            if (!token) {
                console.error('Token not found');
                alert('Authentication token not found. Please log in again.');
                return;
            }
            console.log('Token retrieved:', token); // Debug log

            // Initialize Bootstrap modals
            var addAdminModal = new bootstrap.Modal(document.getElementById('addAdminModal'), { keyboard: false });
            var addVendorModal = new bootstrap.Modal(document.getElementById('addVendorModal'), { keyboard: false });
            var addCustomerModal = new bootstrap.Modal(document.getElementById('addCustomerModal'), { keyboard: false });
            var viewModal = new bootstrap.Modal(document.getElementById('viewModal'), { keyboard: false });
            var editModal = new bootstrap.Modal(document.getElementById('editModal'), { keyboard: false });

            // Trigger add modals manually on button click
            $('.btn-success').on('click', function() {
                addAdminModal.show();
            });
            $('.btn-warning').on('click', function() {
                addVendorModal.show();
            });
            $('.btn-info').on('click', function() {
                addCustomerModal.show();
            });

            // Prevent default modal behavior and handle view manually
            $('.view-btn').on('click', function(e) {
                e.preventDefault(); // Prevent default Bootstrap modal behavior
                const userId = $(this).data('user-id');
                const user = <?php echo json_encode($users); ?>.find(u => u.user_id == userId);
                if (!user) {
                    alert('User not found');
                    return;
                }
                $('#viewEmail').text(user.email || 'N/A');
                $('#viewFirstName').text(user.first_name || 'N/A');
                $('#viewLastName').text(user.last_name || 'N/A');
                $('#viewPhone').text(user.phone || 'N/A');
                $('#viewRole').text(user.role || 'N/A');
                if (user.role === 'vendor') {
                    $('.vendor-fields-view').show();
                    $('#viewVendorName').text(user.vendor_name || 'N/A');
                    $('#viewBio').text(user.bio || 'N/A');
                } else {
                    $('.vendor-fields-view').hide();
                }
                viewModal.show();
                // Reset and hide other modals on view open
                addCustomerModal.hide();
            });

            // Handle edit manually
            $('.edit-btn').on('click', function(e) {
                e.preventDefault(); // Prevent default Bootstrap modal behavior
                const userId = $(this).data('user-id');
                const user = <?php echo json_encode($users); ?>.find(u => u.user_id == userId);
                if (!user) {
                    alert('User not found');
                    return;
                }
                $('#editUserId').val(user.user_id);
                $('#editEmail').val(user.email || '');
                $('#editPassword').val('');
                $('#editFirstName').val(user.first_name || '');
                $('#editLastName').val(user.last_name || '');
                $('#editPhone').val(user.phone || '');
                $('#editRole').val(user.role || 'user');
                if (user.role === 'vendor') {
                    $('.vendor-fields').show();
                    $('#editVendorName').val(user.vendor_name || '');
                    $('#editBio').val(user.bio || '');
                } else {
                    $('.vendor-fields').hide();
                }
                editModal.show();
                // Reset and hide other modals on edit open
                addCustomerModal.hide();
            });

            // Save Edit (submit form)
            $('#editUserForm').on('submit', function(e) {
                e.preventDefault();
                const formData = $(this).serialize();
                $.ajax({
                    url: $(this).attr('action'),
                    method: 'POST',
                    data: formData,
                    success: function(response) {
                        alert(response.success || 'User updated successfully');
                        location.reload();
                    },
                    error: function(xhr) {
                        alert('Failed to update user: ' + xhr.responseText);
                    }
                });
            });

            // Delete functionality
            $('.delete-btn').on('click', function() {
                const userId = $(this).data('id');
                $('#deleteUserId').val(userId);
                $('#deleteModal').modal('show');
            });

            $('#confirmDelete').on('click', function() {
                const userId = $('#deleteUserId').val();
                $.ajax({
                    url: '../../backend/api/v1/users.php?id=' + userId,
                    method: 'DELETE',
                    headers: { 'Authorization': token },
                    success: function(response) {
                        alert(response.message || 'User deleted successfully');
                        location.reload();
                    },
                    error: function(xhr) {
                        console.error('Delete AJAX error:', xhr.status, xhr.responseText);
                        if (xhr.status === 405) {
                            $.ajax({
                                url: '../../backend/api/v1/users.php?id=' + userId,
                                method: 'POST',
                                headers: { 'Authorization': token, 'Content-Type': 'application/json' },
                                data: JSON.stringify({ action: 'delete' }),
                                success: function(response) {
                                    alert(response.message || 'User deleted successfully');
                                    location.reload();
                                },
                                error: function(xhr) {
                                    console.error('POST delete workaround error:', xhr.status, xhr.responseText);
                                    alert('Failed to delete user: ' + xhr.responseText);
                                }
                            });
                        } else {
                            alert('Failed to delete user: ' + xhr.responseText);
                        }
                    }
                });
                $('#deleteModal').modal('hide');
            });

            // Show/hide vendor fields based on role
            $('#editRole').on('change', function() {
                if ($(this).val() === 'vendor') {
                    $('.vendor-fields').show();
                } else {
                    $('.vendor-fields').hide();
                }
            });

            // Handle modal close to prevent unintended triggers
            $('#viewModal').on('hidden.bs.modal', function () {
                addCustomerModal.hide(); // Explicitly hide addCustomerModal on view close
            });
            $('#editModal').on('hidden.bs.modal', function () {
                addCustomerModal.hide(); // Explicitly hide addCustomerModal on edit close
            });
        });
    </script>
</body>
</html>