<?php
session_start();

// Include database connection
require_once '../backend/config/database.php';

// Check if user is authenticated
if (!isset($_SESSION['user_id'])) {
    header('Location: ../login.php');
    exit;
}

$userId = $_SESSION['user_id'];

// Validate product_id
if (!isset($_GET['product_id']) || !is_numeric($_GET['product_id'])) {
    header('Location: index.php?error=Invalid product ID');
    exit;
}

$productId = (int)$_GET['product_id'];

// Initialize database connection
$db = new DatabaseConfig();
$conn = $db->getConnection();

// Function to safely build redirect URL
function buildRedirectUrl($productId, $message, $isError = false) {
    // Get referer or default to index.php
    $referer = $_SERVER['HTTP_REFERER'] ?? 'index.php';
    
    // Security check: ensure referer is from same domain
    $refererHost = parse_url($referer, PHP_URL_HOST);
    $serverHost = $_SERVER['HTTP_HOST'];
    if ($refererHost !== $serverHost) {
        $referer = 'index.php';
    }

    // Parse URL components
    $urlParts = parse_url($referer);
    $path = $urlParts['path'] ?? 'index.php';
    
    // Get existing query parameters
    $query = [];
    if (isset($urlParts['query'])) {
        parse_str($urlParts['query'], $query);
    }
    
    // Add new parameters
    if ($isError) {
        $query['error'] = $message;
    } else {
        $query['message'] = $message;
    }
    $query['scroll_to'] = $productId;
    
    // Build new URL
    $newQuery = http_build_query($query);
    $redirectUrl = $path . '?' . $newQuery;
    
    // Add fragment if exists
    if (isset($urlParts['fragment'])) {
        $redirectUrl .= '#' . $urlParts['fragment'];
    }
    
    return $redirectUrl;
}

try {
    $stmt = $conn->prepare(
        "INSERT INTO cart (user_id, product_id, quantity, added_at) 
        VALUES (:user_id, :product_id, 1, NOW()) 
        ON DUPLICATE KEY UPDATE quantity = quantity + 1"
    );
    $stmt->execute([
        'user_id' => $userId,
        'product_id' => $productId
    ]);

    // Redirect back to original page with success
    $redirectUrl = buildRedirectUrl($productId, 'Added to cart successfully');
    header("Location: $redirectUrl");
    exit;
} catch (PDOException $e) {
    error_log("Failed to add to cart: " . $e->getMessage());
    
    // Redirect back to original page with error
    $redirectUrl = buildRedirectUrl($productId, 'Failed to add to cart', true);
    header("Location: $redirectUrl");
    exit;
}
?>