<?php
session_start();

// Include database connection
require_once './backend/config/database.php';

// Initialize variables for form data and errors
$first_name = $last_name = $email = $phone = $password = $confirm_password = '';
$errors = [];
$success = '';

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Sanitize and validate form data
    $first_name = trim($_POST['first_name'] ?? '');
    $last_name = trim($_POST['last_name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $phone = trim($_POST['phone'] ?? '');
    $password = trim($_POST['password'] ?? '');
    $confirm_password = trim($_POST['confirm_password'] ?? '');

    // Validation
    if (empty($first_name)) {
        $errors[] = "First name is required.";
    }
    if (empty($last_name)) {
        $errors[] = "Last name is required.";
    }
    if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = "A valid email is required.";
    }
    if (empty($phone) || !preg_match('/^0[0-9]{9}$/', $phone)) {
        $errors[] = "Phone number must be 10 digits starting with 0 (e.g., 0791333586).";
    }
    if (empty($password)) {
        $errors[] = "Password is required.";
    }
    if ($password !== $confirm_password) {
        $errors[] = "Passwords do not match.";
    }

    // If no errors, proceed with registration
    if (empty($errors)) {
        $db = new DatabaseConfig();
        $conn = $db->getConnection();

        // Check if email already exists
        $stmt = $conn->prepare("SELECT user_id FROM users WHERE email = :email");
        $stmt->execute(['email' => $email]);
        if ($stmt->fetch()) {
            $errors[] = "Email is already registered.";
        } else {
            // Hash the password
            $hashed_password = password_hash($password, PASSWORD_DEFAULT);

            // Insert user into the database
            $stmt = $conn->prepare(
                "INSERT INTO users (first_name, last_name, email, phone, password, role, created_at) 
                VALUES (:first_name, :last_name, :email, :phone, :password, 'user', NOW())"
            );
            $stmt->execute([
                'first_name' => $first_name,
                'last_name' => $last_name,
                'email' => $email,
                'phone' => $phone,
                'password' => $hashed_password
            ]);

            $success = "Registration successful! You can now <a href='login.php'>log in</a>.";
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Roaming Ndovu Sign-Up Page</title>
    <link rel="icon" type="image/png" href="./logo.png">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/js/all.min.js"></script>
</head>
<body>
    <div class="login-container">
        <div class="icon-placeholder">
            <img src="./logo-bg-removed.png" alt="Roaming Ndovu Logo">
        </div>

        <?php if (!empty($errors)): ?>
            <div class="alert alert-danger">
                <?php foreach ($errors as $error): ?>
                    <p><?php echo htmlspecialchars($error); ?></p>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>

        <?php if (!empty($success)): ?>
            <div class="alert alert-success">
                <?php echo $success; ?>
            </div>
        <?php else: ?>
            <form method="POST" action="">
                <div class="mb-3">
                    <input type="text" name="first_name" class="form-control" placeholder="First Name" value="<?php echo htmlspecialchars($first_name); ?>" required>
                </div>
                <div class="mb-3">
                    <input type="text" name="last_name" class="form-control" placeholder="Last Name" value="<?php echo htmlspecialchars($last_name); ?>" required>
                </div>
                <div class="mb-3">
                    <input type="email" name="email" class="form-control" placeholder="Email" value="<?php echo htmlspecialchars($email); ?>" required>
                </div>
                <div class="mb-3">
                    <input type="tel" name="phone" class="form-control" placeholder="Phone Number" value="<?php echo htmlspecialchars($phone); ?>" required pattern="0[0-9]{9}" title="Phone number must be 10 digits starting with 0 (e.g., 0791333586)">
                </div>
                <div class="mb-3 password-container">
                    <input type="password" name="password" class="form-control" id="password" placeholder="Password" required>
                </div>
                <div class="mb-3 password-container">
                    <input type="password" name="confirm_password" class="form-control" id="confirmPassword" placeholder="Confirm Password" required>
                </div>
                <button type="submit" class="btn btn-login">SIGN UP</button>
            </form>
            <p class="signup-text">Already have an account? <a href="login.php">Login</a></p>
        <?php endif; ?>
    </div>

    <style>
        body {
            background-color: #f5f5f5;
            display: flex;
            justify-content: center;
            align-items: center;
            height: 100vh;
            margin: 0;
            font-family: Arial, sans-serif;
        }

        .login-container {
            background-color: #fff;
            padding: 20px;
            border-radius: 15px;
            box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
            width: 100%;
            max-width: 400px;
            text-align: center;
        }

        .icon-placeholder {
            margin-bottom: 20px;
        }

        .icon-placeholder img {
            width: 100px;
            height: auto;
        }

        .form-control {
            border-radius: 25px;
            padding: 10px 15px;
            border: 1px solid #ccc;
            background-color: #f0f4f8;
            font-size: 14px;
        }

        .form-control:focus {
            border-color: #28a745;
            box-shadow: none;
            background-color: #fff;
        }

        .password-container {
            position: relative;
        }

        .btn-login {
            background-color: #1a3c34;
            color: #fff;
            border: none;
            border-radius: 25px;
            padding: 10px;
            width: 100%;
            font-size: 16px;
            font-weight: bold;
            text-transform: uppercase;
            transition: background-color 0.3s;
        }

        .btn-login:hover {
            background-color: #14532d;
        }

        .signup-text {
            margin-top: 15px;
            font-size: 14px;
            color: #666;
        }

        .signup-text a {
            color: #007bff;
            text-decoration: none;
        }

        .signup-text a:hover {
            text-decoration: underline;
        }

        .alert {
            margin-bottom: 15px;
            border-radius: 10px;
            font-size: 14px;
        }

        .alert-success {
            background-color: #d4edda;
            color: #155724;
        }

        .alert-danger {
            background-color: #f8d7da;
            color: #721c24;
        }

        .alert .btn-close {
            padding: 0.5rem 1rem;
        }
    </style>
</body>
</html>